// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot.subsystems;

import com.ctre.phoenix.ErrorCode;
import com.ctre.phoenix.sensors.Pigeon2_Faults;
import com.ctre.phoenix.sensors.WPI_Pigeon2;
import com.pathplanner.lib.auto.AutoBuilder;
import com.pathplanner.lib.util.HolonomicPathFollowerConfig;
import com.pathplanner.lib.util.PIDConstants;
import com.pathplanner.lib.util.ReplanningConfig;

import edu.wpi.first.math.geometry.Pose2d;
import edu.wpi.first.math.geometry.Pose3d;
import edu.wpi.first.math.geometry.Rotation2d;
import edu.wpi.first.math.geometry.Rotation3d;
import edu.wpi.first.math.geometry.Translation2d;
import edu.wpi.first.math.kinematics.ChassisSpeeds;
import edu.wpi.first.math.kinematics.SwerveDriveKinematics;
import edu.wpi.first.math.kinematics.SwerveDriveOdometry;
import edu.wpi.first.math.kinematics.SwerveModulePosition;
import edu.wpi.first.math.kinematics.SwerveModuleState;
import edu.wpi.first.wpilibj.DriverStation;
import edu.wpi.first.wpilibj.Preferences;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import frc.robot.Constants;

public class DriveSubsystem extends SubsystemBase 
{
  private SwerveDriveKinematics kinematics;
  private SwerveDriveOdometry odometry;
  private SwerveModule[] modules;
  private ChassisSpeeds chassisSpeeds;
  private ChassisSpeeds autoOverrideChassisSpeeds;
  private boolean debug = false;
  private WPI_Pigeon2 pigeon2;
  private SwerveModulePosition[] modulePositions;
  private double maximumLinearSpeed = Constants.DrivetrainConstants.MAX_LINEAR_SPEED;
  private double[] rates = new double[3];
  private boolean parkingBrakeOn = false;

  /** Creates a new DriveSubsystem. */
  public DriveSubsystem()
  {
    pigeon2 = new WPI_Pigeon2(15,"Blue");

    var error = pigeon2.configFactoryDefault();
    if (error != ErrorCode.OK) 
    {
      System.out.println(String.format("PIGEON IMU ERROR: %s", error.toString()));
    }
    error = pigeon2.setYaw(180); //180

    // Make space for four swerve modules:
    modules = new SwerveModule[4];
    modulePositions = new SwerveModulePosition[4];

    //front left
    SwerveModuleIDConfig moduleIDConfig = new SwerveModuleIDConfig(Constants.DrivetrainConstants.FRONT_LEFT_CAN_ID, 
                                                                   Constants.DrivetrainConstants.FRONT_LEFT_STEER_ID, 
                                                                   Constants.DrivetrainConstants.FRONT_LEFT_ENCODER_ID,
                                                                   true,
                                                                   Constants.DrivetrainConstants.DRIVE_TRAIN_CAN_BUS_NAME);

    SwerveModuleConfig moduleConfig = new SwerveModuleConfig(); // Gets preferences and defaults for fields.
    //TODO change the negative and positive Translations till auto works
  
    /*
     * -+   --
     * 
     * ++   +-
     * 
     * 
     * --  -+
     * 
     * +-  --
     * 
     */
    moduleConfig.moduleNumber = 0;
    moduleConfig.position = new Translation2d(Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS), -Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS));
    moduleConfig.steerAngleOffset = Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_FL; //2.879;

    modules[0] = new SwerveModule(moduleConfig, moduleIDConfig);
    modulePositions[0] = new SwerveModulePosition();
    modules[0].invertDriveMotor(true);
    //modules[0].invertSteerMotor(true);
    //modules[0].invertSteerEncoder(true);

    //front right
    moduleIDConfig = new SwerveModuleIDConfig(Constants.DrivetrainConstants.FRONT_RIGHT_CAN_ID, 
                                              Constants.DrivetrainConstants.FRONT_RIGHT_STEER_ID, 
                                              Constants.DrivetrainConstants.FRONT_RIGHT_ENCODER_ID,
                                              false,
                                              Constants.DrivetrainConstants.DRIVE_TRAIN_CAN_BUS_NAME);

    moduleConfig = new SwerveModuleConfig(); // Gets preferences and defaults for fields.
    moduleConfig.moduleNumber = 1;
    moduleConfig.position = new Translation2d(Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS), Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS));
    moduleConfig.steerAngleOffset = Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_FR; // 1.866;

    modules[1] = new SwerveModule(moduleConfig, moduleIDConfig);
    modulePositions[1] = new SwerveModulePosition();
    //modules[1].invertSteerMotor(true);
    //modules[1].invertSteerEncoder(true);

    //back left
    moduleIDConfig = new SwerveModuleIDConfig(Constants.DrivetrainConstants.BACK_LEFT_CAN_ID, 
                                              Constants.DrivetrainConstants.BACK_LEFT_STEER_ID, 
                                              Constants.DrivetrainConstants.BACK_LEFT_ENCODEr_ID,
                                              true,
                                              Constants.DrivetrainConstants.DRIVE_TRAIN_CAN_BUS_NAME);

    moduleConfig = new SwerveModuleConfig(); // Gets preferences and defaults for fields.
    moduleConfig.moduleNumber = 2;
    moduleConfig.position = new Translation2d(-Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS), -Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS));
    moduleConfig.steerAngleOffset = Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_BL; // 2.422;

    modules[2] = new SwerveModule(moduleConfig, moduleIDConfig);
    modulePositions[2] = new SwerveModulePosition();
    modules[2].invertDriveMotor(true);
    //modules[2].invertSteerMotor(true);
    //modules[2].invertSteerEncoder(true);

    //back right
    moduleIDConfig = new SwerveModuleIDConfig(Constants.DrivetrainConstants.BACK_RIGHT_CAN_ID, 
                                              Constants.DrivetrainConstants.BACK_RIGHT_STEER_ID, 
                                              Constants.DrivetrainConstants.BACK_RIGHT_ENCODER_ID,
                                              false,
                                              Constants.DrivetrainConstants.DRIVE_TRAIN_CAN_BUS_NAME);
                   
                                              
    moduleConfig = new SwerveModuleConfig(); // Gets preferences and defaults for fields.
    moduleConfig.moduleNumber = 3;
    moduleConfig.position = new Translation2d(-Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS), Preferences.getDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS));
    moduleConfig.steerAngleOffset = Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_BR; //1.109;

    

    modules[3] = new SwerveModule(moduleConfig, moduleIDConfig);
    modulePositions[3] = new SwerveModulePosition();
    //modules[3].invertSteerMotor(true);
    //modules[3].invertSteerEncoder(true);

    // Create our kinematics class
    kinematics = new SwerveDriveKinematics(
      modules[0].position,
      modules[1].position,
      modules[2].position,
      modules[3].position
    );


    // Create odometry:
    modules[0].updatePosition(modulePositions[0]);
    modules[1].updatePosition(modulePositions[1]);
    modules[2].updatePosition(modulePositions[2]);
    modules[3].updatePosition(modulePositions[3]);
    odometry = new SwerveDriveOdometry(kinematics, Rotation2d.fromDegrees(getHeading()), modulePositions, new Pose2d(0,0,new Rotation2d(Math.PI)));

    // Configure maximum linear speed for limiting:
    //maximumLinearSpeed = Preferences.getDouble("Drive.MaximumLinearSpeed", Constants.DrivetrainConstants.MAX_LINEAR_SPEED_PREF);
    maximumLinearSpeed = Constants.DrivetrainConstants.MAX_LINEAR_SPEED;
    // Initial chassis speeds are zero:
    chassisSpeeds = new ChassisSpeeds(0,0,0);

    // Configure AutoBuilder
    AutoBuilder.configureHolonomic(
            this::getPose, // Robot pose supplier
            this::resetPose, // Method to reset odometry (will be called if your auto has a starting pose)
            this::getChassisSpeeds, // ChassisSpeeds supplier. MUST BE ROBOT RELATIVE
            this::setChassisSpeeds, // Method that will drive the robot given ROBOT RELATIVE ChassisSpeeds
            new HolonomicPathFollowerConfig( // HolonomicPathFollowerConfig, this should likely live in your Constants class
                    new PIDConstants(0.001, 0.0, 0.0), // Translation PID constants
                    new PIDConstants(0.001, 0.0, 0.0), // Rotation PID constants
                    4.5, // Max module speed, in m/s
                    0.37, // Drive base radius in meters. Distance from robot center to furthest module.
                    new ReplanningConfig() // Default path replanning config. See the API for the options here
            ),
            () -> {
              // Boolean supplier that controls when the path will be mirrored for the red alliance
              // This will flip the path being followed to the red side of the field.
              // THE ORIGIN WILL REMAIN ON THE BLUE SIDE

              var alliance = DriverStation.getAlliance();
              if (alliance.isPresent()) {
                return alliance.get() == DriverStation.Alliance.Red;
              }
              return false;
            },
            this // Reference to this subsystem to set requirements
    );

  }

/*// Odometry class for tracking robot pose
  SwerveDriveOdometry autoOdometry = new SwerveDriveOdometry(
    Constants.DrivetrainConstants.kDriveKinematics,
    Rotation2d.fromDegrees(pigeon2.getAngle()),
    new SwerveModulePosition[] {
        modules[0].getModulePositions(),
        modules[1].getModulePositions(),
        modules[2].getModulePositions(),
        modules[3].getModulePositions()
    });*/

  /////////////testing auto
  public Pose2d getPose(){
    return odometry.getPoseMeters();
  }
//test if making the gyro negative here changes the rotation to be correct for pathplanner
  public void resetPose(Pose2d pose) {
    odometry.resetPosition(
            Rotation2d.fromDegrees(pigeon2.getAngle()),
            new SwerveModulePosition[] {
                modules[0].getModulePositions(),
                modules[1].getModulePositions(),
                modules[2].getModulePositions(),
                modules[3].getModulePositions()
            },
            pose);
}

// Set the commanded chassis speeds for the drive subsystem.
  public void setSpeeds(ChassisSpeeds speeds)
  {
    chassisSpeeds = ChassisSpeeds.discretize(speeds, 0.02);
  }

  public void driveRobotRelative(ChassisSpeeds speeds){
    drive(speeds, false);
}

public void drive(ChassisSpeeds speeds,boolean fieldRelative){
  if(fieldRelative){
      speeds = ChassisSpeeds.fromFieldRelativeSpeeds(speeds, getPose().getRotation());
}
  var swerveModuleStates = Constants.DrivetrainConstants.kDriveKinematics.toSwerveModuleStates(speeds);
  SwerveDriveKinematics.desaturateWheelSpeeds(swerveModuleStates, Constants.DrivetrainConstants.MAX_LINEAR_SPEED);
  setModuleStates(swerveModuleStates);
}

public void setModuleStates(SwerveModuleState[] desiredStates) {
  SwerveDriveKinematics.desaturateWheelSpeeds(
      desiredStates, Constants.DrivetrainConstants.MAX_LINEAR_SPEED);
  modules[0].setDesiredState(desiredStates[0]);
  modules[1].setDesiredState(desiredStates[1]);
  modules[2].setDesiredState(desiredStates[2]);
  modules[3].setDesiredState(desiredStates[3]);
}















  // Initialize preferences for this class:
  public static void initPreferences() 
  {
    Preferences.initDouble("Drive.Module0.SteerAngleOffset", Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_FL); // Radians. // 2.879
    Preferences.initDouble("Drive.Module1.SteerAngleOffset", Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_FR);//1.866
    Preferences.initDouble("Drive.Module2.SteerAngleOffset", Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_BL);//2.422
    Preferences.initDouble("Drive.Module3.SteerAngleOffset", Constants.DrivetrainConstants.STEER_ANGLE_OFFSET_MOD_BR);//1.189
    Preferences.initDouble("Drive.MaximumLinearSpeed", Constants.DrivetrainConstants.MAX_LINEAR_SPEED_PREF); // Meters/second
    Preferences.initDouble("Drive.ModulePositions", Constants.DrivetrainConstants.MODULE_POSITIONS);
  }

  public String getDiagnostics() 
  {
    String result = modules[0].getDiagnostics();
    result += modules[1].getDiagnostics();
    result += modules[2].getDiagnostics();
    result += modules[3].getDiagnostics();

    Pigeon2_Faults faults = new Pigeon2_Faults();
    pigeon2.getFaults(faults);
    if(faults.hasAnyFault()){
      result += faults.toString();
    }
    ErrorCode error = pigeon2.clearStickyFaults(500);
    if (error != ErrorCode.OK) {
        result += String.format(" Cannot contact the pigeon.");
    }
    
    //Check errors for all hardware
    return result;
  }

  public void setDebugMode(boolean debug) 
  {
    this.debug = debug;
  }

  //Returns IMU heading in degrees
  public double getHeading() 
  {
    return pigeon2.getYaw();
  }

  // Wraps the heading
  public double getWrappedHeading()
  {
    double heading = getHeading() % 360;

    if(heading >= 0)
    {
      return heading;
    }

    else
    {
      return 360 + heading;
    }
  }

  public double getPitch()
  {
    return pigeon2.getPitch();
  }

  public double getRoll()
  {
    return pigeon2.getRoll();
  }

  public double getPitchRate()
  {
    pigeon2.getRawGyro(rates);
    return rates[1];
  }

  // Reset IMU heading to zero degrees
  public void zeroHeading() 
  {
    pigeon2.setYaw(180);
  }

  // Set the commanded chassis speeds for the drive subsystem.
  public void setChassisSpeeds(ChassisSpeeds speeds)
  {
    chassisSpeeds = speeds;
  }

  public void setAutoOverrideChassisSpeeds(ChassisSpeeds autoOverrideChassisSpeeds) {
    this.autoOverrideChassisSpeeds = autoOverrideChassisSpeeds;
  }

  // Return the measured chassis speeds for the drive subsystem.
  public ChassisSpeeds getChassisSpeeds()
  {
    SwerveModuleState[] wheelStates = new SwerveModuleState[4];
    wheelStates[0] = new SwerveModuleState();
    wheelStates[1] = new SwerveModuleState();
    wheelStates[2] = new SwerveModuleState();
    wheelStates[3] = new SwerveModuleState();

    wheelStates[0].speedMetersPerSecond = modules[0].getDriveVelocity();
    wheelStates[1].speedMetersPerSecond = modules[1].getDriveVelocity();
    wheelStates[2].speedMetersPerSecond = modules[2].getDriveVelocity();
    wheelStates[3].speedMetersPerSecond = modules[3].getDriveVelocity();

    wheelStates[0].angle = new Rotation2d(modules[0].getSteeringAngle());
    wheelStates[1].angle = new Rotation2d(modules[1].getSteeringAngle());
    wheelStates[2].angle = new Rotation2d(modules[2].getSteeringAngle());
    wheelStates[3].angle = new Rotation2d(modules[3].getSteeringAngle());

    return kinematics.toChassisSpeeds(wheelStates);
  }

  // puts the motors in brake mode
  public void setBrakes(boolean brakeOn)
  {
    modules[0].setBrake(brakeOn);
    modules[1].setBrake(brakeOn);
    modules[2].setBrake(brakeOn);
    modules[3].setBrake(brakeOn);
  }

  // returns the wheel positions
  public SwerveDriveKinematics getKinematics()
  {
    return kinematics;
  }

  public void updateOdometry() 
  {
    modules[0].updatePosition(modulePositions[0]);
    modules[1].updatePosition(modulePositions[1]);
    modules[2].updatePosition(modulePositions[2]);
    modules[3].updatePosition(modulePositions[3]);
    odometry.update(Rotation2d.fromDegrees(getHeading()), modulePositions);
  }

  /*public void resetOdometry(Pose2d where)
  {
    odometry.resetPosition(Rotation2d.fromDegrees(getHeading()), modulePositions, where);
  }*/

  public Pose2d getOdometry()
  {
    return new Pose2d(odometry.getPoseMeters().getX(), odometry.getPoseMeters().getY(), Rotation2d.fromDegrees(getHeading()));
  }

  public double getOdometryX(){
    return odometry.getPoseMeters().getX();
  }

  public double getOdometryY(){
    return odometry.getPoseMeters().getY();
  }

  public void resetPosePathPlanner(Pose2d pose) {
    odometry.resetPosition(pigeon2.getRotation2d(), getPositions(), pose);
  }

  public SwerveModulePosition[] getPositions() {
    SwerveModulePosition[] positions = new SwerveModulePosition[modules.length];
    for (int i = 0; i < modules.length; i++) {
      positions[i] = modules[i].getModulePositions();
    }
    return positions;
  }

  /**
   * Basic simulation of a swerve module, will just hold its current state and not use any hardware
   */
  class SimSwerveModule {
    private SwerveModulePosition currentPosition = new SwerveModulePosition();
    private SwerveModuleState currentState = new SwerveModuleState();

    public SwerveModulePosition getPosition() {
      return currentPosition;
    }

    public SwerveModuleState getState() {
      return currentState;
    }

    public void setTargetState(SwerveModuleState targetState) {
      // Optimize the state
      currentState = SwerveModuleState.optimize(targetState, currentState.angle);

      currentPosition = new SwerveModulePosition(currentPosition.distanceMeters + (currentState.speedMetersPerSecond * 0.02), currentState.angle);
    }
  }

  public Pose3d get3dOdometry()
  {
    // return odometry position as a pose 3d
    Pose2d odo = getOdometry();
    //TODO: use internal roll and pitch methods later
    return new Pose3d(odo.getX(), odo.getY(), 0.0, new Rotation3d(pigeon2.getRoll(), pigeon2.getPitch(), getHeading()));
  }
  
  // This method will be called once per scheduler run
  @Override
  public void periodic()
  {
    if (!debug && !parkingBrakeOn) //disables motors when parking brakes are active
    {
      // Add in chassis speed values from autonomous or auto-assist functions while allowing unspecified values to contiue to be applied by controller. 
      if (autoOverrideChassisSpeeds != null) {
        if (autoOverrideChassisSpeeds.vxMetersPerSecond != 0) {
          chassisSpeeds.vxMetersPerSecond = autoOverrideChassisSpeeds.vxMetersPerSecond;
        }
        if (autoOverrideChassisSpeeds.vyMetersPerSecond != 0) {
          chassisSpeeds.vyMetersPerSecond = autoOverrideChassisSpeeds.vyMetersPerSecond;
        }
        if (autoOverrideChassisSpeeds.omegaRadiansPerSecond != 0) {
          chassisSpeeds.omegaRadiansPerSecond = autoOverrideChassisSpeeds.omegaRadiansPerSecond;
        }
      }
      
      SwerveModuleState[] states = kinematics.toSwerveModuleStates(chassisSpeeds);
      SwerveDriveKinematics.desaturateWheelSpeeds(states, maximumLinearSpeed);
      states[0] = optimizeB(states[0], new Rotation2d(modules[0].getSteeringAngle()));
      states[1] = optimizeB(states[1], new Rotation2d(modules[1].getSteeringAngle()));
      states[2] = optimizeB(states[2], new Rotation2d(modules[2].getSteeringAngle()));
      states[3] = optimizeB(states[3], new Rotation2d(modules[3].getSteeringAngle()));

      modules[0].setCommand(states[0].angle.getRadians(), states[0].speedMetersPerSecond);
      modules[1].setCommand(states[1].angle.getRadians(), states[1].speedMetersPerSecond);
      modules[2].setCommand(states[2].angle.getRadians(), states[2].speedMetersPerSecond);
      modules[3].setCommand(states[3].angle.getRadians(), states[3].speedMetersPerSecond);

      SmartDashboard.putNumber("Actual Module speed", modules[0].getDriveVelocity());
      SmartDashboard.putNumber("Actual Module angle", modules[0].getSteeringAngle());
      SmartDashboard.putNumber("RAW Actual Module angle - FL", modules[0].getRawSteeringAngle());
      SmartDashboard.putNumber("RAW Actual Module angle - FR", modules[1].getRawSteeringAngle());
      SmartDashboard.putNumber("RAW Actual Module angle - BR", modules[2].getRawSteeringAngle());
      SmartDashboard.putNumber("RAW Actual Module angle - BL", modules[3].getRawSteeringAngle());
      SmartDashboard.putNumber("Commanded Speed", states[0].speedMetersPerSecond);
      SmartDashboard.putNumber("Commanded angle", states[0].angle.getRadians());

    }
    else if(!parkingBrakeOn)
    { //in debug mode
      SmartDashboard.putNumber("Module 0 Velocity in ticks", modules[0].getDriveRawVelocity());
    }
    updateOdometry();
    SmartDashboard.putNumber("Odometry.X", odometry.getPoseMeters().getX());
    SmartDashboard.putNumber("Odometry.Y", odometry.getPoseMeters().getY());
    SmartDashboard.putNumber("Odometry.Heading", this.getHeading());
    SmartDashboard.putNumber("Odometry.Wrapped.Heading", this.getWrappedHeading());

    SmartDashboard.putNumber("Pitch", getPitch());
    SmartDashboard.putNumber("Roll", getRoll());
    SmartDashboard.putNumber("Rate", getPitchRate());
  }


  // rotates all the wheels to be facing inwards and stops the motors to hold position
  public void parkingBrake(boolean parkingBrakeOn) 
  {
    this.parkingBrakeOn = parkingBrakeOn;
    if (parkingBrakeOn)
    {
      SwerveModuleState[] states = kinematics.toSwerveModuleStates(chassisSpeeds);
      states[0] = optimizeB(new SwerveModuleState(0, new Rotation2d(Math.PI / 4)), new Rotation2d(modules[0].getSteeringAngle()));
      states[1] = optimizeB(new SwerveModuleState(0, new Rotation2d(-Math.PI / 4)), new Rotation2d(modules[1].getSteeringAngle()));
      states[2] = optimizeB(new SwerveModuleState(0, new Rotation2d(-Math.PI / 4)), new Rotation2d(modules[2].getSteeringAngle()));
      states[3] = optimizeB(new SwerveModuleState(0, new Rotation2d(Math.PI / 4)), new Rotation2d(modules[3].getSteeringAngle()));

      modules[0].setCommand(states[0].angle.getRadians(), states[0].speedMetersPerSecond);
      modules[1].setCommand(states[1].angle.getRadians(), states[1].speedMetersPerSecond);
      modules[2].setCommand(states[2].angle.getRadians(), states[2].speedMetersPerSecond);
      modules[3].setCommand(states[3].angle.getRadians(), states[3].speedMetersPerSecond);
    }
  }

  public boolean getParkingBrake()
  {
    return parkingBrakeOn;
  }

  public void setDebugSpeed(double speed) // sets the speed directly
  {
    modules[0].setDriveVelocity(speed);
    modules[1].setDriveVelocity(speed);
    modules[2].setDriveVelocity(speed);
    modules[3].setDriveVelocity(speed);
  }

  public void setDebugAngle(double angle) // sets the angle directly
  {
    SmartDashboard.putNumber("Debug Angle", angle);

    modules[0].setSteerAngle(angle);
    modules[1].setSteerAngle(angle);
    modules[2].setSteerAngle(angle);
    modules[3].setSteerAngle(angle);
  }

  public void setDebugDrivePower(double power) // sets the power directly
  {
    modules[0].setDebugTranslate(power);
    modules[1].setDebugTranslate(power);
    modules[2].setDebugTranslate(power);
    modules[3].setDebugTranslate(power);
  }


  // optimizes the module states to take the shortest path to the desired position
  public static SwerveModuleState optimizeB(SwerveModuleState desiredState, Rotation2d currentAngle)
  {
    double targetAngle = placeInAppropriate0To360Scope(currentAngle.getDegrees(), desiredState.angle.getDegrees());
    double targetSpeed = desiredState.speedMetersPerSecond;
    double delta = targetAngle - currentAngle.getDegrees();
    if (Math.abs(delta) > 90)
    {
      targetSpeed = -targetSpeed;
      targetAngle = delta > 90 ? (targetAngle -= 180) : (targetAngle += 180);
    }        
    return new SwerveModuleState(targetSpeed, Rotation2d.fromDegrees(targetAngle));
  }

  /**
   * @param scopeReference Current Angle
   * @param newAngle Target Angle
   * @return Closest angle within scope
   */
  

  // places the desired angle to be in a 0 to 360 scope to minimize distance for wheels to rotate 
  private static double placeInAppropriate0To360Scope(double scopeReference, double newAngle) 
  {
    double lowerBound;
    double upperBound;
    double lowerOffset = scopeReference % 360;
    if (lowerOffset >= 0)
    {
      lowerBound = scopeReference - lowerOffset;
      upperBound = scopeReference + (360 - lowerOffset);
    } 
    else
    {
      upperBound = scopeReference - lowerOffset;
      lowerBound = scopeReference - (360 + lowerOffset);
    }
    while (newAngle < lowerBound)
    {
      newAngle += 360;
    }
    while (newAngle > upperBound)
    {
      newAngle -= 360;
    }
    if (newAngle - scopeReference > 180)
    {
      newAngle -= 360;
    } 
    else if (newAngle - scopeReference < -180) 
    {
      newAngle += 360;
    }
    return newAngle;
  }
}