package frc.robot.subsystems;

import java.io.File;
import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;

import edu.wpi.first.math.geometry.Pose2d;
import edu.wpi.first.math.geometry.Rotation2d;
import edu.wpi.first.math.geometry.Transform2d;
import edu.wpi.first.wpilibj.DriverStation;
import edu.wpi.first.wpilibj.DriverStation.Alliance;
import edu.wpi.first.wpilibj.Filesystem;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import frc.robot.Constants.ReefFace;
import frc.robot.Constants.ReefHeight;
import frc.robot.Constants.ReefSide;
import frc.robot.subsystems.vision.AprilTags;

/** This subsystem is a dedicated place for state variables that control how the robot behaves.
 * While they could technically live in existing subsystems, some of them (like the coral delivery
 * variables) involve multiple subsystems in the pieces of functionality they control, so putting
 * them in one would be an arbitrary choice.
 */
public class RobotControl extends SubsystemBase {

  public enum RobotControlMode {
    CORAL, ALGAE, CLIMB, MANUAL
  }

  public enum DriveMode {
    NORMAL, SLOW
  }

  public enum AprilTagMode {
    AUTO, MANUAL
  }

  public enum AlgaeDeliveryMode {
    PROCESSOR, BARGE
  }

  public enum AlgaeIntakeMode {
    LOWER, UPPER
  }

  private RobotControlMode controlMode = RobotControlMode.CORAL;

  private DriveMode driveMode = DriveMode.NORMAL;

  private ReefFace coralDeliveryFace = ReefFace.CLOSE;
  private ReefSide coralDeliverySide = ReefSide.LEFT;
  private ReefHeight coralDeliveryHeight = ReefHeight.L1;

  private AlgaeIntakeMode algaeIntakeMode = AlgaeIntakeMode.LOWER;
  private AlgaeDeliveryMode algaeDeliveryMode = AlgaeDeliveryMode.BARGE;

  private Map<Alliance, Map<ReefFace, Integer>> allianceDeliveryPosToAprilTag = Map.of(
    Alliance.Red, Map.of(
      ReefFace.CLOSE, 7,
      ReefFace.CLOSELEFT, 6,
      ReefFace.CLOSERIGHT, 8,
      ReefFace.FARLEFT, 11,
      ReefFace.FARRIGHT, 9,
      ReefFace.FAR, 10
    ),
    Alliance.Blue, Map.of(
      ReefFace.CLOSE, 18,
      ReefFace.CLOSELEFT, 19,
      ReefFace.CLOSERIGHT, 17,
      ReefFace.FARLEFT, 20,
      ReefFace.FARRIGHT, 22,
      ReefFace.FAR, 21
    )
  );
  private Pose2d autoDeliveryPose = new Pose2d();
  private Map<Alliance, Map<ReefFace, Map<ReefSide, String>>> allianceDeliveryPositionToPathName = Map.of(
    Alliance.Red, Map.of(
      ReefFace.CLOSE, Map.of(
        ReefSide.LEFT, "",
        ReefSide.RIGHT, ""
      ),
      ReefFace.CLOSELEFT, Map.of(
        ReefSide.LEFT, "",
        ReefSide.RIGHT, ""
      ),
      ReefFace.CLOSERIGHT, Map.of(
        ReefSide.LEFT, "AlignRedCRL",
        ReefSide.RIGHT, ""
      ),
      ReefFace.FARLEFT, Map.of(
        ReefSide.LEFT, "",
        ReefSide.RIGHT, ""
      ),
      ReefFace.FARRIGHT, Map.of(
        ReefSide.LEFT, "",
        ReefSide.RIGHT, ""
      ),
      ReefFace.FAR, Map.of(
        ReefSide.LEFT, "",
        ReefSide.RIGHT, ""
      )
    ),
    Alliance.Blue, Map.of(
      ReefFace.CLOSE, Map.of(
        ReefSide.LEFT, "AlignBlueCCL",
        ReefSide.RIGHT, "AlignBlueCCL"
      ),
      ReefFace.CLOSELEFT, Map.of(
        ReefSide.LEFT, "AlignBlueCLL",
        ReefSide.RIGHT, "AlignBlueCLR"
      ),
      ReefFace.CLOSERIGHT, Map.of(
        ReefSide.LEFT, "AlignBlueCRL",
        ReefSide.RIGHT, "AlignBlueCRR"
      ),
      ReefFace.FARLEFT, Map.of(
        ReefSide.LEFT, "AlignBlueFLL",
        ReefSide.RIGHT, "AlignBlueFLR"
      ),
      ReefFace.FARRIGHT, Map.of(
        ReefSide.LEFT, "AlignBlueFRL",
        ReefSide.RIGHT, "AlignBlueFRR"
      ),
      ReefFace.FAR, Map.of(
        ReefSide.LEFT, "AlignBlueFCL",
        ReefSide.RIGHT, "AlignBlueFCR"
      )
    )
  );

  private AprilTagMode aprilTagMode = AprilTagMode.MANUAL;

  @Override
  public void periodic() {
    SmartDashboard.putString("RobotControl/AprilTagMode", getAprilTagMode().toString());
    SmartDashboard.putString("RobotControl/ControlMode", getControlMode().toString());
    SmartDashboard.putString("RobotControl/DriveMode", getDriveMode().toString());
    SmartDashboard.putString("RobotControl/ReefFace", getCoralDeliveryFace().toString());
    SmartDashboard.putString("RobotControl/ReefSide", getCoralDeliverySide().toString());
    SmartDashboard.putString("RobotControl/ReefHeight", getCoralDeliveryHeight().toString());
    SmartDashboard.putString("RobotControl/AlgaeDeliveryMode", getAlgaeDeliveryMode().toString());
    SmartDashboard.putString("RobotControl/AlgaeIntakeMode", getAlgaeIntakeMode().toString());
  }

  /** Gets the current control mode */
  public RobotControlMode getControlMode() {
    return this.controlMode;
  }

  /** Rotate the robot control mode through the four available
   * values in the following order: Coral, Algae, Climb, Manual */
  public Command rotateRobotControlMode() {
    return runOnce(() -> {
      if (this.controlModeIsCoral()) {
        this.controlMode = RobotControlMode.ALGAE;
      } else if (this.controlModeIsAlgae()) {
        this.controlMode = RobotControlMode.CLIMB;
      } else if (this.controlModeIsClimb()) {
        this.controlMode = RobotControlMode.MANUAL;
      } else {
        this.controlMode = RobotControlMode.CORAL;
      }
    });
  }

  /** Set the robot control mode to Coral */
  public Command setControlModeCoral() {
    return runOnce(() -> this.controlMode = RobotControlMode.CORAL).withName("SetControlModeCoral");
  }

  /** Returns whether the current control mode is Coral */
  public boolean controlModeIsCoral() {
    return this.controlMode == RobotControlMode.CORAL;
  }

  /** Set the robot control mode to Algae */
  public Command setControlModeAlgae() {
    return runOnce(() -> this.controlMode = RobotControlMode.ALGAE).withName("SetControlModeAlgae");
  }

  /** Returns whether the current control mode is Algae */
  public boolean controlModeIsAlgae() {
    return this.controlMode == RobotControlMode.ALGAE;
  }

  /** Set the robot control mode to Climb */
  public Command setControlModeClimb() {
    return runOnce(() -> this.controlMode = RobotControlMode.CLIMB).withName("SetControlModeClimb");
  }

  /** Returns whether the current control mode is Climb */
  public boolean controlModeIsClimb() {
    return this.controlMode == RobotControlMode.CLIMB;
  }

  /** Set the robot control mode to Manual */
  public Command setControlModeManual() {
    return runOnce(() -> this.controlMode = RobotControlMode.MANUAL).withName("SetControlModeManual");
  }

  /** Returns whether the current control mode is Manual */
  public boolean controlModeIsManual() {
    return this.controlMode == RobotControlMode.MANUAL;
  }

  /** Gets the current drive mode */
  public DriveMode getDriveMode() {
    return this.driveMode;
  }

  /** Toggle the robot drive mode between Normal and Slow drive modes */
  public Command toggleDriveMode() {
    return runOnce(() -> {
      if (this.driveMode == DriveMode.NORMAL) {
        this.driveMode = DriveMode.SLOW;
      } else {
        this.driveMode = DriveMode.NORMAL;
      }
    }).withName("ToggleDriveMode");
  }

  /** Sets the robot drive mode to Slow for as long as the command
   * is running, setting it back to Normal when the command exits.
   */
  public Command holdSlowDriveMode() {
    return startEnd(
      () -> this.driveMode = DriveMode.SLOW,
      () -> this.driveMode = DriveMode.NORMAL
    ).withName("HoldSlowDriveMode");
  }

  /** Gets the coral delivery face */
  public ReefFace getCoralDeliveryFace() {
    return this.coralDeliveryFace;
  }

  /** Gets the coral delivery side */
  public ReefSide getCoralDeliverySide() {
    return this.coralDeliverySide;
  }

  /** Gets the coral delivery height */
  public ReefHeight getCoralDeliveryHeight() {
    return this.coralDeliveryHeight;
  }

  /** Set the automatic coral delivery to target the specified height of a reef branch */
  public Command setCoralDeliveryHeight(ReefHeight newHeight) {
    return runOnce(() -> this.coralDeliveryHeight = newHeight).withName("SetCoralDeliveryHeight");
  }

  /** Set the automatic coral delivery to target the specified reef face and side */
  public Command setCoralDeliveryFaceAndSide(ReefFace newFace, ReefSide newSide) {
    return runOnce(() -> {
      this.coralDeliveryFace = newFace;
      this.coralDeliverySide = newSide;
      Optional<Alliance> curAlliance = DriverStation.getAlliance();
      if (curAlliance.isPresent()) {
        this.autoDeliveryPose = AprilTags
          .GetAprilTag(allianceDeliveryPosToAprilTag.get(curAlliance.get()).get(this.coralDeliveryFace)).pose2d
            .transformBy(new Transform2d(.45, this.coralDeliverySide == ReefSide.LEFT ? -0.1 : 0.1,
              Rotation2d.fromDegrees(180)));
      }
    }).withName("SetCoralDeliveryFaceAndSide");
  }

  /** Gets the pose we would navigate to during auto coral delivery mode, based
   * on the current coral delivery face and side
   */
  public Pose2d getAutoDeliveryPose() {
    return this.autoDeliveryPose;
  }

  public String getAutoDeliveryPathFilepath() {
    Optional<Alliance> curAlliance = DriverStation.getAlliance();
    if (curAlliance.isPresent()) {
      String pathName = this.allianceDeliveryPositionToPathName.get(curAlliance.get()).get(this.coralDeliveryFace)
        .get(this.coralDeliverySide);
      String path = Path
        .of(Filesystem.getDeployDirectory() + File.separator + "pathplanner" + File.separator + "paths" + File.separator
          + pathName + ".path")
        .toString();
      return path;
    } else {
      return "NotFound";
    }
  }

  /** Sets the AprilTag delivery mode (either Auto or Manual) */
  public Command setAprilTagMode(AprilTagMode newMode) {
    return runOnce(() -> this.aprilTagMode = newMode).withName("SetAprilTagMode");
  }

  /** Toggles the AprilTag delivery mode between Auto and Manual */
  public Command toggleAprilTagMode() {
    return runOnce(() -> {
      if (this.aprilTagMode == AprilTagMode.AUTO) {
        this.aprilTagMode = AprilTagMode.MANUAL;
      } else {
        this.aprilTagMode = AprilTagMode.AUTO;
      }
    }).withName("ToggleAprilTagMode");
  }

  /** Gets the current AprilTag delivery mode */
  public AprilTagMode getAprilTagMode() {
    return this.aprilTagMode;
  }

  /** Returns whether the current AprilTag delivery mode is Auto */
  public boolean aprilTagModeIsAuto() {
    return this.aprilTagMode == AprilTagMode.AUTO;
  }

  /** Returns whether the current AprilTag delivery mode is Manual */
  public boolean aprilTagModeIsManual() {
    return this.aprilTagMode == AprilTagMode.MANUAL;
  }

  /** Sets the algae intake mode to lower (between L2 and L3) */
  public Command setAlgaeIntakeModeLower() {
    return runOnce(() -> this.algaeIntakeMode = AlgaeIntakeMode.LOWER);
  }

  /** Sets the algae intake mode to upper (between L3 and L4) */
  public Command setAlgaeIntakeModeUpper() {
    return runOnce(() -> this.algaeIntakeMode = AlgaeIntakeMode.UPPER);
  }

  /** Sets the algae delivery mode to barge */
  public Command setAlgaeDeliveryModeBarge() {
    return runOnce(() -> this.algaeDeliveryMode = AlgaeDeliveryMode.BARGE);
  }

  /** Sets the algae delivery mode to processor */
  public Command setAlgaeDeliveryModeProcessor() {
    return runOnce(() -> this.algaeDeliveryMode = AlgaeDeliveryMode.PROCESSOR);
  }

  /** Gets the current algae delivery mode */
  public AlgaeDeliveryMode getAlgaeDeliveryMode() {
    return this.algaeDeliveryMode;
  }

  /** Gets the current algae intake mode */
  public AlgaeIntakeMode getAlgaeIntakeMode() {
    return this.algaeIntakeMode;
  }
}
