// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot.subsystems;

import static edu.wpi.first.units.Units.Degrees;
import static edu.wpi.first.units.Units.Rotations;
import static edu.wpi.first.units.Units.RotationsPerSecond;
import static edu.wpi.first.units.Units.RotationsPerSecondPerSecond;
import static edu.wpi.first.units.Units.Second;

import java.util.Set;
import java.util.function.Supplier;

import com.ctre.phoenix6.configs.CurrentLimitsConfigs;
import com.ctre.phoenix6.configs.FeedbackConfigs;
import com.ctre.phoenix6.configs.MotionMagicConfigs;
import com.ctre.phoenix6.configs.Slot0Configs;
import com.ctre.phoenix6.configs.TalonFXConfiguration;
import com.ctre.phoenix6.configs.TalonFXConfigurator;
import com.ctre.phoenix6.controls.DutyCycleOut;
import com.ctre.phoenix6.controls.DynamicMotionMagicVoltage;
import com.ctre.phoenix6.controls.MotionMagicVoltage;
import com.ctre.phoenix6.hardware.CANcoder;
import com.ctre.phoenix6.hardware.TalonFX;
import com.ctre.phoenix6.signals.NeutralModeValue;

import edu.wpi.first.math.MathUtil;
import edu.wpi.first.wpilibj.DigitalInput;

import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.DeferredCommand;
import edu.wpi.first.wpilibj2.command.FunctionalCommand;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import frc.robot.Constants;
import frc.robot.DashboardLinkedNumber;
import frc.robot.RobotContainer;
import frc.robot.TalonUtilities;

public class Manipulator extends SubsystemBase {
  private static TalonFX wheels_motor = new TalonFX(Constants.CB_IDS.MANIPULATOR_WHEELS_ID,
    Constants.CB_NAMES.UpperArm);
  private static TalonFXConfiguration wheels_motor_cfg = new TalonFXConfiguration()
    .withCurrentLimits(new CurrentLimitsConfigs()
      .withStatorCurrentLimit(120)
      .withStatorCurrentLimitEnable(true))
    .withMotionMagic(new MotionMagicConfigs()
      .withMotionMagicCruiseVelocity(RotationsPerSecond.of(20)) // 1 (mechanism) rotation per second cruise
      .withMotionMagicAcceleration(RotationsPerSecondPerSecond.of(20)) // Take approximately 0.5 seconds to reach max vel
      .withMotionMagicJerk(RotationsPerSecondPerSecond.per(Second).of(200))) // Take approximately 0.1 seconds to reach max accel
    .withSlot0(new Slot0Configs()
      .withKS(0)
      .withKV(0)
      .withKA(0)
      .withKP(60)
      .withKI(0)
      .withKD(0));
  private static TalonFX manipulator_angle_motor = new TalonFX(Constants.CB_IDS.MANIPULATOR_ANGLE_ID,
    Constants.CB_NAMES.UpperArm);
  private static CANcoder manipulator_angle_encoder = new CANcoder(Constants.CB_IDS.MANIPULATOR_ANGLE_CANCODER_ID,
    Constants.CB_NAMES.UpperArm);
  private static TalonFXConfiguration manipulator_angle_motor_cfg = new TalonFXConfiguration()
    .withFeedback(new FeedbackConfigs().withRemoteCANcoder(manipulator_angle_encoder))
    .withCurrentLimits(new CurrentLimitsConfigs()
      .withStatorCurrentLimit(120)
      .withStatorCurrentLimitEnable(true))
    .withMotionMagic(new MotionMagicConfigs()
      .withMotionMagicCruiseVelocity(RotationsPerSecond.of(2)) // 1 (mechanism) rotation per second cruise
      .withMotionMagicAcceleration(RotationsPerSecondPerSecond.of(50)) // Take approximately 0.5 seconds to reach max vel
      .withMotionMagicJerk(RotationsPerSecondPerSecond.per(Second).of(1500))) // Take approximately 0.1 seconds to reach max accel
    .withSlot0(new Slot0Configs()
      .withKS(0)
      .withKV(0)
      .withKA(0)
      .withKP(60)
      .withKI(0)
      .withKD(0.1));
  // private static long last_linebreak_time = -1; // -1 when it isn't tripped
  private static DigitalInput coral_linebreak_sensor = new DigitalInput(
    Constants.DIGITAL_INPUT_IDS.MANIPULATOR_LINEBREAK_ID);
  private double manualManipulatorAngleTarget = 0;

  private DashboardLinkedNumber kS = new DashboardLinkedNumber(0, "ManipulatorSubsystem/kS", this::applyPIDs);
  private DashboardLinkedNumber kV = new DashboardLinkedNumber(0, "ManipulatorSubsystem/kV", this::applyPIDs);
  private DashboardLinkedNumber kA = new DashboardLinkedNumber(0, "ManipulatorSubsystem/kA", this::applyPIDs);
  private DashboardLinkedNumber kP = new DashboardLinkedNumber(60, "ManipulatorSubsystem/kP", this::applyPIDs);
  private DashboardLinkedNumber kI = new DashboardLinkedNumber(0, "ManipulatorSubsystem/kI", this::applyPIDs);
  private DashboardLinkedNumber kD = new DashboardLinkedNumber(0, "ManipulatorSubsystem/kD", this::applyPIDs);

  public Manipulator() {
    TalonUtilities.applyTalonFXConfiguration(wheels_motor, wheels_motor_cfg);
    wheels_motor.setNeutralMode(NeutralModeValue.Brake);
    TalonUtilities.applyTalonFXConfiguration(manipulator_angle_motor, manipulator_angle_motor_cfg);
  }

  @Override
  public void periodic() {
    SmartDashboard.putNumber("ManipulatorSubsystem/ManipulatorAngle", getManipulatorAngle());
    SmartDashboard.putNumber("ManipulatorSubsystem/WheelSpeed", getWheelSpeed());
    SmartDashboard.putBoolean("ManipulatorSubsystem/LinbreakTripped", getLinebreakIsTripped());
    // if (this.getLinebreakIsTripped()) {
    //   if (last_linebreak_time != 0) {
    //     last_linebreak_time = System.currentTimeMillis();
    //   }
    // } else {
    //   last_linebreak_time = -1;
    // }
  }

  private void applyPIDs(double newVal) {
    manipulator_angle_motor_cfg.Slot0 = new Slot0Configs()
      .withKS(this.kS.get())
      .withKV(this.kV.get())
      .withKA(this.kA.get())
      .withKP(this.kP.get())
      .withKI(this.kI.get())
      .withKD(this.kD.get());
    TalonFXConfigurator configurator = manipulator_angle_motor.getConfigurator();
    configurator.apply(manipulator_angle_motor_cfg);
  }

  /** Set the manipulator angle to go to (and maintain) a target angle, in degrees, and exit the command once the angle
   * reaches within the angleProximity parameter (in degrees) of the target angle. If the angleProximity parameter is
   * negative the command will exit immediately */
  public Command setManipulatorAngle(double angle, double angleProximity) {
    final double target_angle = MathUtil.clamp(angle, Constants.ManipulatorConstants.MinManipulatorAngle,
      Constants.ManipulatorConstants.MaxManipulatorAngle);
    return new DeferredCommand(
      () -> new FunctionalCommand(
        () -> manipulator_angle_motor
          .setControl(new DynamicMotionMagicVoltage(
            Degrees.of(target_angle).in(Rotations),
            RobotContainer.robotcontrol.controlModeIsAlgae() ? 1 : 2,
            50,
            1500
          ).withSlot(0)),
        () -> {},
        interrupted -> {},
        () -> angleProximity < 0 ? true : Math.abs(getManipulatorAngle() - target_angle) < angleProximity),
      Set.of(this, RobotContainer.robotcontrol)
    ).withName("SetManipulatorAngle");
  }

  /** Set the manipulator angle to go to (and maintain) a target angle, in degrees, and exit the command once the angle
   * reaches within the default manipulator angle proximity (in degrees) of the target angle */
  public Command setManipulatorAngle(double angle) {
    return this.setManipulatorAngle(angle, Constants.ManipulatorConstants.ManipulatorAngleDefaultProximity);
  }

  /** Adjust the manipulator angle manually */
  public Command adjustManipulatorAngleManual(Supplier<Double> power) {
    return new FunctionalCommand(
      () -> this.manualManipulatorAngleTarget = getManipulatorAngle(),
      () -> {
        this.manualManipulatorAngleTarget = MathUtil.clamp(this.manualManipulatorAngleTarget + (2 * power.get()),
          Constants.ManipulatorConstants.MinManipulatorAngle, Constants.ManipulatorConstants.MaxManipulatorAngle);
        manipulator_angle_motor
          .setControl(new MotionMagicVoltage(Degrees.of(this.manualManipulatorAngleTarget).in(Rotations)).withSlot(0));
      },
      interrupted -> {
        manipulator_angle_motor.setNeutralMode(NeutralModeValue.Brake);
      },
      () -> false,
      this
    ).withName("AdjustManipulatorAngleManual");
  }

  /** Set the manipulator angle to change at a particular speed. Stops automatically when angle exceeds limits */
  public Command setManipulatorAngleSpeed(Supplier<Double> power) {
    return runEnd(
      () -> {
        double target_power = MathUtil.clamp(power.get(), -1.0, 1.0)
          * Constants.ManipulatorConstants.MaxManipulatorAngleSpeed;
        manipulator_angle_motor.setControl(new DutyCycleOut(target_power));
      },
      () -> manipulator_angle_motor.stopMotor()
    ).withName("SetManipulatorAngleSpeed");
  }

  /** Stops the manipulator angle motors from moving */
  public Command stopManipulatorAngleMotor() {
    return runOnce(() -> manipulator_angle_motor.stopMotor()).withName("StopManipulatorAngleMotor");
  }

  /** Actively tries to stop the wheels from turning at all, in an attempt to hold onto the algae */
  public Command holdAlgae() {
    return runOnce(() -> {
      var curPos = wheels_motor.getRotorPosition().getValue().in(Degrees);
      wheels_motor.setControl(new MotionMagicVoltage(Degrees.of(curPos).in(Rotations)).withSlot(0));
    });
  }

  /** Set the Manipulator wheels to rotate at a particular speed. Positive will intake coral, negative will output */
  public Command setWheelSpeed(double speed) {
    double target_speed = MathUtil.clamp(speed, -1, 1) * Constants.ManipulatorConstants.MaxWheelsSpeed;
    return runOnce(() -> wheels_motor.setControl(new DutyCycleOut(target_speed))).withName("SetWheelSpeed");
  }

  /** Sets the manipulator wheels to rotate at a particular speed */
  public Command holdWheelSpeed(double speed) {
    double target_speed = MathUtil.clamp(speed, -1, 1) * Constants.ManipulatorConstants.MaxWheelsSpeed;
    return startEnd(
      () -> wheels_motor.setControl(new DutyCycleOut(target_speed)),
      () -> {}
    ).withName("HoldWheelSpeed");
  }

  /** Sets the manipulator wheels to rotate at a particular speed, stopping when the Command is cancelled */
  public Command holdWheelSpeedAndEnd(double speed) {
    double target_speed = MathUtil.clamp(speed, -1, 1) * Constants.ManipulatorConstants.MaxWheelsSpeed;
    return startEnd(
      () -> wheels_motor.setControl(new DutyCycleOut(target_speed)),
      () -> wheels_motor.stopMotor()
    ).withName("HoldWheelSpeed");
  }

  /** Stop the manipulator wheels from moving */
  public Command stopWheels() {
    return runOnce(() -> wheels_motor.stopMotor());
  }

  /** Runs the wheels inwards until the linebreak sensor trips, indicating that a coral is present */
  public Command intakeCoral() {
    return holdWheelSpeed(0.6)
      .until(this::getLinebreakIsTripped)
      .withName("IntakeCoral");
  }

  /** Get the angle of the wheel motor in degrees */
  public double getManipulatorAngle() {
    return manipulator_angle_motor.getPosition().getValue().in(Degrees);
  }

  /** Get the speed of the wheel motor in rotations per second */
  public double getWheelSpeed() {
    return wheels_motor.getVelocity().getValue().in(RotationsPerSecond);
  }

  /** Get whether the wheels are moving */
  public boolean getWheelsAreMoving() {
    return Math.abs(getWheelSpeed()) > 0;
  }

  /** Get whether the linebreak sensor is tripped (indicating coral is present)*/
  public boolean getLinebreakIsTripped() {
    return !coral_linebreak_sensor.get();
  }

  /** Set the manipulator angle motor to Brake mode in neutral */
  public Command setManipulatorAngleBrake() {
    return runOnce(() -> manipulator_angle_motor.setNeutralMode(NeutralModeValue.Brake));
  }

  /** Set the manipulator angle motor to Coast mode in neutral */
  public void setManipulatorAngleCoast() {
    manipulator_angle_motor.setNeutralMode(NeutralModeValue.Coast);
  }

  /** Get the current value of neutral mode for the manipulator angle motor */
  public String getManipulatorAngleBrakeMode() {
    return manipulator_angle_motor.getBridgeOutput().getValue().toString();
  }

  /** Sets the wheels motor to brake mode when not otherwise being controlled */
  public Command setWheelsMotorBrakeMode() {
    return runOnce(() -> wheels_motor.setNeutralMode(NeutralModeValue.Brake));
  }
}
