// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot.subsystems;

import static edu.wpi.first.units.Units.*;

import java.util.function.Supplier;

import com.ctre.phoenix6.configs.CurrentLimitsConfigs;
import com.ctre.phoenix6.configs.FeedbackConfigs;
import com.ctre.phoenix6.configs.MotionMagicConfigs;
import com.ctre.phoenix6.configs.MotorOutputConfigs;
import com.ctre.phoenix6.configs.Slot0Configs;
import com.ctre.phoenix6.configs.TalonFXConfiguration;
import com.ctre.phoenix6.controls.DutyCycleOut;
import com.ctre.phoenix6.controls.DynamicMotionMagicVoltage;
import com.ctre.phoenix6.controls.Follower;
import com.ctre.phoenix6.controls.MotionMagicVoltage;
import com.ctre.phoenix6.hardware.CANcoder;
import com.ctre.phoenix6.hardware.TalonFX;
import com.ctre.phoenix6.signals.NeutralModeValue;

import edu.wpi.first.math.MathUtil;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.FunctionalCommand;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import frc.robot.Constants;
import frc.robot.TalonUtilities;

public class Lift extends SubsystemBase {
  // Shared motor config
  private CurrentLimitsConfigs currentlimit_cfg = new CurrentLimitsConfigs()
    .withStatorCurrentLimit(120)
    .withStatorCurrentLimitEnable(true);
  // Motors that drive the lift elevator position
  private TalonFX lift_motor = new TalonFX(Constants.CB_IDS.LIFT_HEIGHT_LEFT_ID, Constants.CB_NAMES.Drivetrain);
  private TalonFX lift_motor_follower = new TalonFX(Constants.CB_IDS.LIFT_HEIGHT_RIGHT_ID,
    Constants.CB_NAMES.Drivetrain);
  private TalonFXConfiguration lift_motor_cfg = new TalonFXConfiguration()
    .withMotorOutput(new MotorOutputConfigs()
      .withNeutralMode(NeutralModeValue.Brake))
    .withCurrentLimits(currentlimit_cfg)
    .withMotionMagic(new MotionMagicConfigs() // This is overridden by the DynamicMotionMagicVoltage below
      .withMotionMagicCruiseVelocity(RotationsPerSecond.of(80))
      .withMotionMagicAcceleration(RotationsPerSecondPerSecond.of(160))
      .withMotionMagicJerk(RotationsPerSecondPerSecond.per(Second).of(1600)))
    .withSlot0(new Slot0Configs()
      .withKS(0)
      .withKV(0)
      .withKA(0)
      .withKP(4.8)
      .withKI(0)
      .withKD(0.1));
  private double manualLiftPosTarget = 0;

  // Motors and sensors that drive the lift angle
  private TalonFX lift_angle_motor = new TalonFX(Constants.CB_IDS.LIFT_ANGLE_LEFT_ID,
    Constants.CB_NAMES.Drivetrain);
  private TalonFX lift_angle_motor_follower = new TalonFX(Constants.CB_IDS.LIFT_ANGLE_RIGHT_ID,
    Constants.CB_NAMES.Drivetrain);
  private CANcoder lift_angle_encoder = new CANcoder(Constants.CB_IDS.LIFT_ANGLE_CANCODER_ID,
    Constants.CB_NAMES.Drivetrain);
  private // Lift Angle Motors Config
  TalonFXConfiguration lift_angle_motor_cfg = new TalonFXConfiguration()
    .withMotorOutput(new MotorOutputConfigs()
      .withNeutralMode(NeutralModeValue.Brake))
    // Get feedback from the CANCoder instead of the internal rotor sensor
    .withFeedback(new FeedbackConfigs().withRemoteCANcoder(lift_angle_encoder))
    .withCurrentLimits(currentlimit_cfg)
    .withMotionMagic(new MotionMagicConfigs()
      .withMotionMagicCruiseVelocity(RotationsPerSecond.of(1)) // 0.5 (mechanism) rotations per second cruise
      .withMotionMagicAcceleration(RotationsPerSecondPerSecond.of(25))
      .withMotionMagicJerk(RotationsPerSecondPerSecond.per(Second).of(200)))
    .withSlot0(new Slot0Configs()
      .withKS(0)
      .withKV(0)
      .withKA(0)
      .withKP(60)
      .withKI(0.1)
      .withKD(0.1));
  private double manualLiftAngleTarget = 0;

  // private DashboardLinkedNumber liftkS = new DashboardLinkedNumber(0, "LiftSubsystem/liftkS", this::applyLiftPIDs);
  // private DashboardLinkedNumber liftkV = new DashboardLinkedNumber(0, "LiftSubsystem/liftkV", this::applyLiftPIDs);
  // private DashboardLinkedNumber liftkA = new DashboardLinkedNumber(0, "LiftSubsystem/liftkA", this::applyLiftPIDs);
  // private DashboardLinkedNumber liftkP = new DashboardLinkedNumber(3, "LiftSubsystem/liftkP", this::applyLiftPIDs);
  // private DashboardLinkedNumber liftkI = new DashboardLinkedNumber(0, "LiftSubsystem/liftkI", this::applyLiftPIDs);
  // private DashboardLinkedNumber liftkD = new DashboardLinkedNumber(0.1, "LiftSubsystem/liftkD", this::applyLiftPIDs);

  // private DashboardLinkedNumber anglekS = new DashboardLinkedNumber(0, "LiftSubsystem/anglekS", this::applyAnglePIDs);
  // private DashboardLinkedNumber anglekV = new DashboardLinkedNumber(0, "LiftSubsystem/anglekV", this::applyAnglePIDs);
  // private DashboardLinkedNumber anglekA = new DashboardLinkedNumber(0, "LiftSubsystem/anglekA", this::applyAnglePIDs);
  // private DashboardLinkedNumber anglekP = new DashboardLinkedNumber(60, "LiftSubsystem/anglekP", this::applyAnglePIDs);
  // private DashboardLinkedNumber anglekI = new DashboardLinkedNumber(0.1, "LiftSubsystem/anglekI", this::applyAnglePIDs);
  // private DashboardLinkedNumber anglekD = new DashboardLinkedNumber(0.1, "LiftSubsystem/anglekD", this::applyAnglePIDs);

  public Lift() {
    TalonUtilities.applyTalonFXConfiguration(lift_motor, lift_motor_cfg);
    lift_motor_follower.setControl(new Follower(lift_motor.getDeviceID(), true));
    TalonUtilities.applyTalonFXConfiguration(lift_angle_motor, lift_angle_motor_cfg);
    lift_angle_motor_follower.setControl(new Follower(lift_angle_motor.getDeviceID(), true));
  }

  @Override
  public void periodic() {
    SmartDashboard.putNumber("LiftSubsystem/Lift POS", getLiftPosition());
    SmartDashboard.putNumber("LiftSubsystem/Lift Angle", getLiftAngle());
  }

  // private void applyAnglePIDs(double newVal) {
  //   lift_angle_motor_cfg.Slot0 = new Slot0Configs()
  //     .withKS(this.anglekS.get())
  //     .withKV(this.anglekV.get())
  //     .withKA(this.anglekA.get())
  //     .withKP(this.anglekP.get())
  //     .withKI(this.anglekI.get())
  //     .withKD(this.anglekD.get());
  //   TalonFXConfigurator configurator = lift_angle_motor.getConfigurator();
  //   configurator.apply(lift_angle_motor_cfg);
  // }

  // private void applyLiftPIDs(double newVal) {
  //   lift_motor_cfg.Slot0 = new Slot0Configs()
  //     .withKS(this.liftkS.get())
  //     .withKV(this.liftkV.get())
  //     .withKA(this.liftkA.get())
  //     .withKP(this.liftkP.get())
  //     .withKI(this.liftkI.get())
  //     .withKD(this.liftkD.get());
  //   TalonFXConfigurator configurator = lift_motor.getConfigurator();
  //   configurator.apply(lift_motor_cfg);
  // }

  //region Lift Angle Commands
  /** Set the lift angle to go to (and maintain) a target angle, in degrees, and exit the command once the lift angle
   * reaches within the angleProximity parameter (in degrees) of the target angle. If the angleProximity parameter is
   * negative the command will exit immediately */
  public Command setLiftAngle(double angle, double angleProximity) {
    final double target_angle = MathUtil.clamp(angle, Constants.LiftConstants.MinLiftAngle,
      Constants.LiftConstants.MaxLiftAngle);
    return new FunctionalCommand(
      () -> lift_angle_motor.setControl(new MotionMagicVoltage(Degrees.of(target_angle).in(Rotations)).withSlot(0)),
      () -> {},
      interrupted -> {},
      () -> angleProximity < 0 ? true : Math.abs(getLiftAngle() - target_angle) < angleProximity,
      this
    ).withName("SetLiftAngle");
  }

  /** Set the lift angle to go to (and maintain) a target angle, in degrees, and exit the command once the lift angle
   * reaches within the default lift angle proximity (in degrees) of the target angle */
  public Command setLiftAngle(double angle) {
    return this.setLiftAngle(angle, Constants.LiftConstants.LiftAngleDefaultProximity);
  }

  /** Adjust the lift angle manually */
  public Command adjustLiftAngleManual(Supplier<Double> power) {
    return new FunctionalCommand(
      () -> this.manualLiftAngleTarget = getLiftAngle(),
      () -> {
        this.manualLiftAngleTarget = MathUtil.clamp(this.manualLiftAngleTarget + (2 * power.get()),
          Constants.LiftConstants.MinLiftAngle, Constants.LiftConstants.MaxLiftAngle);
        lift_angle_motor
          .setControl(new MotionMagicVoltage(Degrees.of(this.manualLiftAngleTarget).in(Rotations)).withSlot(0));
      },
      interrupted -> {
        lift_angle_motor.setNeutralMode(NeutralModeValue.Brake);
        lift_angle_motor_follower.setNeutralMode(NeutralModeValue.Brake);
      },
      () -> false,
      this
    ).withName("AdjustLiftAngleManual");
  }

  /** Set the lift angle to change at a certain speed. Stops automatically when angle exceeds limits */
  public Command setLiftAngleSpeed(Supplier<Double> power) {
    return runEnd(
      () -> {
        double target_power = MathUtil.clamp(power.get(), -1.0, 1.0) * Constants.LiftConstants.MaxLiftAngleRate;
        lift_angle_motor.setControl(new DutyCycleOut(target_power));
      },
      () -> lift_angle_motor.stopMotor()
    ).withName("SetLiftAngleSpeed");
  }

  /** Stop the lift angle motors from moving */
  public Command stopLiftAngleMotors() {
    return runOnce(() -> lift_angle_motor.stopMotor()).withName("StopLiftAngleMotors");
  }

  /** Set the lift angle motors to 'Brake' mode */
  public Command setLiftAngleBrake() {
    return runOnce(() -> {
      lift_angle_motor.setNeutralMode(NeutralModeValue.Brake);
      lift_angle_motor_follower.setNeutralMode(NeutralModeValue.Brake);
    }).withName("SetLiftAngleBrake");
  }

  /** Set the lift angle motors to 'Coast' mode */
  public Command setLiftAngleCoast() {
    return runOnce(() -> {
      lift_angle_motor.setNeutralMode(NeutralModeValue.Coast);
      lift_angle_motor_follower.setNeutralMode(NeutralModeValue.Coast);
    }).withName("SetLiftAngleCoast");
  }
  //endregion

  //region Lift Position Commands
  /** Set the lift position to go to (and maintain) a target angle, in degrees, and exit the command once the lift position
   * reaches within the posProximity parameter (in degrees) of the target position. If the posProximity parameter is negative
   * the command will exit immediately */
  public Command setLiftPosition(double pos, double posProximity) {
    final double target_pos = MathUtil.clamp(pos, Constants.LiftConstants.MinLiftPos,
      Constants.LiftConstants.MaxLiftPos);
    return new FunctionalCommand(
      () -> {
        double curPos = getLiftPosition();
        boolean goingUp = target_pos > curPos;
        lift_motor.setControl(new DynamicMotionMagicVoltage(
          Degrees.of(target_pos).in(Rotations),
          goingUp ? 100 : 48,
          goingUp ? 180 : 100,
          goingUp ? 1800 : 1000
        ).withSlot(0));
        setLiftBrake();
      },
      () -> {},
      interrupted -> {},
      () -> posProximity < 0 ? true : Math.abs(getLiftPosition() - target_pos) < posProximity,
      this
    ).withName("SetLiftPosition");
  }

  /** Set the lift position to go to (and maintain) a target position, in degrees, and exit the command once the lift position
   * reaches within the default lift position proximity (in degrees) of the target angle */
  public Command setLiftPosition(double pos) {
    return this.setLiftPosition(pos, Constants.LiftConstants.LiftPosDefaultProximity);
  }

  public Command setLiftPositionBargeScore(double pos, double posProximity) {
    final double target_pos = MathUtil.clamp(pos, Constants.LiftConstants.MinLiftPos,
      Constants.LiftConstants.MaxLiftPos);
    return new FunctionalCommand(
      () -> {
        lift_motor.setControl(new DynamicMotionMagicVoltage(
          Degrees.of(target_pos).in(Rotations),
          100,
          200,
          2000
        ).withSlot(0));
        setLiftBrake();
      },
      () -> {},
      interrupted -> {},
      () -> posProximity < 0 ? true : Math.abs(getLiftPosition() - target_pos) < posProximity,
      this
    ).withName("SetLiftPositionBargeScore");
  }

  /** Adjust the lift position manually */
  public Command adjustLiftPosManual(Supplier<Double> power) {
    return new FunctionalCommand(
      () -> this.manualLiftPosTarget = getLiftPosition(),
      () -> {
        this.manualLiftPosTarget = MathUtil.clamp(this.manualLiftPosTarget + (25 * power.get()),
          Constants.LiftConstants.MinLiftPos, Constants.LiftConstants.MaxLiftPos);
        lift_motor
          .setControl(new MotionMagicVoltage(Degrees.of(this.manualLiftPosTarget).in(Rotations)).withSlot(0));
      },
      interrupted -> {
        lift_motor.setNeutralMode(NeutralModeValue.Brake);
        lift_motor_follower.setNeutralMode(NeutralModeValue.Brake);
      },
      () -> false,
      this
    ).withName("AdjustLiftPosManual");
  }

  /** Set the lift to move at a certain speed. Stops automatically when lift position exceeds limits */
  public Command setLiftSpeed(Supplier<Double> power) {
    return runEnd(
      () -> {
        double target_power = MathUtil.clamp(power.get(), -1.0, 1.0) * Constants.LiftConstants.MaxLiftRate;
        lift_motor.setControl(new DutyCycleOut(target_power));
      },
      () -> lift_motor.stopMotor()
    ).withName("SetLiftSpeed");
  }

  /** Stop the lift motors from moving */
  public Command stopLiftMotors() {
    return runOnce(() -> lift_motor.stopMotor()).withName("StopLiftMotors");
  }

  /** Set the lift angle motors to 'Brake' mode */
  public Command setLiftBrake() {
    return runOnce(() -> {
      lift_motor.setNeutralMode(NeutralModeValue.Brake);
      lift_motor_follower.setNeutralMode(NeutralModeValue.Brake);
    }).withName("SetLiftBrake");
  }

  /** Set the lift angle motors to 'Coast' mode */
  public Command setLiftCoast() {
    return runOnce(() -> {
      lift_motor.setNeutralMode(NeutralModeValue.Coast);
      lift_motor_follower.setNeutralMode(NeutralModeValue.Coast);
    }).withName("SetLiftCoast");
  }
  //endregion

  /** Get the neutral mode of the lift angle motors */
  public String getLiftAngleNeutralMode() {
    return lift_angle_motor.getBridgeOutput().getValue().toString();
  }

  /** Get the neutral mode of the lift motors */
  public String getLiftNeutralMode() {
    return lift_motor.getBridgeOutput().getValue().toString();
  }

  /** Get the current height of the lift, in Degrees */
  public double getLiftPosition() {
    return lift_motor.getPosition().getValue().in(Degrees);
  }

  /** Get the current speed of the lift height change, in rotations per second */
  public double getLiftPositionSpeed() {
    return lift_motor.getVelocity().getValue().in(RotationsPerSecond);
  }

  /** Gets the current angle of the lift, in degrees */
  public double getLiftAngle() {
    return lift_angle_motor.getPosition().getValue().in(Degrees);
  }

  /** Gets the current angular velocity of the lift, in rotations per second */
  public double getLiftAngleSpeed() {
    return lift_angle_motor.getVelocity().getValue().in(RotationsPerSecond);
  }
}
