// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot.subsystems;

import static edu.wpi.first.units.Units.*;

import java.util.Map;
import java.util.Optional;

import edu.wpi.first.wpilibj.AddressableLED;
import edu.wpi.first.wpilibj.AddressableLEDBuffer;
import edu.wpi.first.wpilibj.AddressableLEDBufferView;
import edu.wpi.first.wpilibj.DriverStation;
import edu.wpi.first.wpilibj.DriverStation.Alliance;
import edu.wpi.first.wpilibj.LEDPattern;
import edu.wpi.first.wpilibj.util.Color;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.SubsystemBase;
import frc.robot.Constants;
import frc.robot.RobotContainer;
import frc.robot.subsystems.RobotControl.RobotControlMode;

public class LEDs extends SubsystemBase {
  private final AddressableLED led = new AddressableLED(Constants.LEDConstants.LED_PORT);
  private final AddressableLEDBuffer buffer = new AddressableLEDBuffer(Constants.LEDConstants.LED_WIDTH);

  private final AddressableLEDBufferView seg1 = new AddressableLEDBufferView(buffer, 0, 17);
  private final AddressableLEDBufferView seg2 = new AddressableLEDBufferView(buffer, 18, 53);
  private final AddressableLEDBufferView seg3 = new AddressableLEDBufferView(buffer, 54, 89);
  private final AddressableLEDBufferView seg4 = new AddressableLEDBufferView(buffer, 90, 125);
  private final AddressableLEDBufferView seg5 = new AddressableLEDBufferView(buffer, 126, 143);

  private Color allyColor;

  public LEDs() {
    led.setLength(buffer.getLength());
    led.start();
    setAllianceColor();
  }

  @Override
  public void periodic() {
    led.setData(buffer);
  }

  private void setAllianceColor() {
    Optional<Alliance> ally = DriverStation.getAlliance();
    if (ally.isPresent()) {
      if (ally.get() == Alliance.Red) {
        allyColor = Color.kRed;
      } else {
        allyColor = Color.kBlue;
      }
    } else { // This should not happen; if it does, then our Driver's station isn't connected
      allyColor = Color.kDarkMagenta;
    }
  }

  /** Set the LEDs to display the patterns used to display the robot's current mode */
  public Command showModePattern() {
    return run(() -> {
      setAllianceColor();
      RobotControlMode controlMode = RobotContainer.robotcontrol.getControlMode();
      Color controlColor;
      if (controlMode == RobotControlMode.CORAL) {
        controlColor = Color.kDarkGray;
      } else if (controlMode == RobotControlMode.ALGAE) {
        controlColor = Color.kGreen;
      } else if (controlMode == RobotControlMode.CLIMB) {
        controlColor = allyColor;
      } else {
        controlColor = Color.kBlack;
      }
      LEDPattern.solid(allyColor).applyTo(seg1);
      if (controlMode == RobotControlMode.CLIMB) {
        LEDPattern.steps(Map.of(0, controlColor, 0.25, Color.kBlack)).scrollAtRelativeSpeed(Percent.per(Second).of(100))
          .applyTo(seg2);
      } else {
        LEDPattern.solid(controlColor).applyTo(seg2);
      }
      LEDPattern.solid(allyColor).applyTo(seg3);
      if (controlMode == RobotControlMode.CLIMB) {
        LEDPattern.steps(Map.of(0, controlColor, 0.25, Color.kBlack)).scrollAtRelativeSpeed(Percent.per(Second).of(100))
          .applyTo(seg4);
      } else {
        LEDPattern.solid(controlColor).applyTo(seg4);
      }
      LEDPattern.solid(allyColor).applyTo(seg5);
    });
  }

  /** Set a new pattern to be displayed on the LEDs */
  private Command runPattern(LEDPattern pattern) {
    return run(() -> {
      setAllianceColor();
      pattern.applyTo(buffer);
    }).ignoringDisable(true);
  }

  /** Set the LEDs to a solid pattern in the color of the team we're on */
  public Command breathAllyColor() {
    return LED_Breathe(allyColor);
  }

  /** Set the LEDs to a solid pattern in the color of the team we're on */
  public Command solidAllyColor() {
    return LED_Color(allyColor);
  }

  /** Set the LEDs to display a solid pattern in the specified color */
  public Command LED_Color(Color color) {
    return runPattern(LEDPattern.solid(color)).withName("LEDColor");
  }

  /** Set the LEDs to display a breathing pattern with a particular solid color */
  public Command LED_Breathe(Color color) {
    return runPattern(LEDPattern
      .solid(color)
      .breathe(Seconds.of(4)))
        .withName("LEDBreathe");
  }

  /** Set the LEDs to display a rainbow pattern that scrolls the length of the LED strip */
  public Command LED_Rainbow() {
    return runPattern(LEDPattern
      .rainbow(255, 128)
      .scrollAtAbsoluteSpeed(MetersPerSecond.of(1), Meters.of(1 / 120)))
        .withName("LEDRainbow");
  }
}
