package frc.robot.commands;

import static edu.wpi.first.units.Units.Seconds;

import java.util.Map;
import java.util.Set;

import com.pathplanner.lib.auto.AutoBuilder;
import com.pathplanner.lib.path.PathConstraints;

import edu.wpi.first.math.util.Units;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.DeferredCommand;
import edu.wpi.first.wpilibj2.command.InstantCommand;
import edu.wpi.first.wpilibj2.command.ParallelCommandGroup;
import edu.wpi.first.wpilibj2.command.SelectCommand;
import edu.wpi.first.wpilibj2.command.SequentialCommandGroup;
import edu.wpi.first.wpilibj2.command.WaitCommand;
import frc.robot.Constants;
import frc.robot.RobotContainer;
import frc.robot.Constants.ReefHeight;
import frc.robot.subsystems.Lift;
import frc.robot.subsystems.Manipulator;
import frc.robot.subsystems.RobotControl;
import frc.robot.subsystems.vision.LiftCamera;

public class CoralCommands {

  private Lift lift;
  private Manipulator manipulator;
  private RobotControl robotcontrol;
  private LiftCamera liftCamera;

  public CoralCommands(Lift lift, Manipulator manipulator, RobotControl robotcontrol, LiftCamera liftCamera) {
    this.lift = lift;
    this.manipulator = manipulator;
    this.robotcontrol = robotcontrol;
    this.liftCamera = liftCamera;
  }

  /** Drive until the robot is in front of and correctly offset from the correct AprilTag for
   * the selected reef face and side then move the arm to be at the correct delivery height */
  public Command deliverToAprilTag() {
    return new SequentialCommandGroup(
      // new DeferredCommand(
      //   () -> {
      //     System.out.println("Executing deliverToAprilTag command");
      //     PathPlannerPath path;
      //     try {
      //       path = PathPlannerPath.fromPathFile(robotcontrol.getAutoDeliveryPathFilepath());
      //     }
      //     catch (Exception e) {
      //       System.out.println("Error loading Pathplanner path file");
      //       path = null;
      //     }
      //     if (path == null) {
      //       System.out.println("Path is null");
      //       return new InstantCommand();
      //     } else {
      //       System.out.println("Executing automatic pathfinding");
      //       return AutoBuilder.pathfindThenFollowPath(path,
      //         new PathConstraints(1, 3.0, Units.degreesToRadians(540), Units.degreesToRadians(720)));
      //     }
      //   },
      //   Set.of(RobotContainer.drivetrain)
      // )
      new DeferredCommand(
        () -> AutoBuilder.pathfindToPose(robotcontrol.getAutoDeliveryPose(),
          new PathConstraints(4.5, 6.0, Units.degreesToRadians(540), Units.degreesToRadians(720)),
          0.0),
        Set.of(RobotContainer.drivetrain, RobotContainer.robotcontrol)
      ),
      // new DeferredCommand(
      //   () -> {
      //     Pose2d targetPose = robotcontrol.getAutoDeliveryPose();
      //     Pose2d currentPose = RobotContainer.drivetrain.getState().Pose;
      //     return RobotContainer.drivetrain.applyRequest(() -> new SwerveRequest.FieldCentricFacingAngle()
      //       .withTargetDirection(targetPose.getRotation())
      //       .withVelocityX((targetPose.getX() - currentPose.getX()) / .25)
      //       .withVelocityY((targetPose.getY() - currentPose.getY()) / .25)
      //     );
      //   },
      //   Set.of(RobotContainer.drivetrain)),
      // new WaitCommand(Seconds.of(.25)),
      // new StartEndCommand(() -> RobotContainer.drivetrain.setControl(new SwerveRequest.Idle()), () -> {}),
      // new ParallelCommandGroup(
      //   new FunctionalCommand(
      //     () -> {},
      //     () -> {
      //       Pose2d targetPose = robotcontrol.getAutoDeliveryPose();
      //       Pose2d currentPose = RobotContainer.drivetrain.getState().Pose;
      //       Transform2d diff = targetPose.minus(currentPose);
      //       RobotContainer.drivetrain.setControl(new SwerveRequest.FieldCentricFacingAngle()
      //         .withTargetDirection(targetPose.getRotation())
      //         .withVelocityX(diff.getX())
      //         .withVelocityY(diff.getY()));
      //     },
      //     interrupted -> {
      //       RobotContainer.drivetrain.setControl(new SwerveRequest.Idle());
      //     },
      //     () -> {
      //       Pose2d targetPose = robotcontrol.getAutoDeliveryPose();
      //       Pose2d currentPose = RobotContainer.drivetrain.getState().Pose;
      //       Transform2d diff = targetPose.minus(currentPose);
      //       return diff.getX() < .01 && diff.getY() < .01 && diff.getRotation().getDegrees() < 1;
      //     }, RobotContainer.drivetrain).withTimeout(Seconds.of(.5)),
      goToSelectedCoralDeliveryHeight()
    // )
    ).withName("DeliverToAprilTag");
  }

  /** Intake a coral from the intake station. If allowed to execute fully this command will not
   * exit until the coral is in the manipulator and the lift has gone back to the storage position */
  public Command intakeCoralFromStation() {
    return new SequentialCommandGroup(
      lift.setLiftBrake(),
      manipulator.setWheelSpeed(0.6),
      lift.setLiftAngle(-32, -1),
      manipulator.setManipulatorAngle(Constants.ManipulatorConstants.MinManipulatorAngle, -1),
      lift.setLiftPosition(2260.0),
      // The following three commands can be commented out if the manipulator's linebreak sensor isn't working
      manipulator.intakeCoral(),
      new WaitCommand(0.4),
      goToStoragePosition()
    ).withName("IntakeCoralFromStation");
  }

  /** Intake a coral from the intake station, but a coral's width away. Otherwise identical to the above command */
  public Command intakeCoralFromStationCoralDist() {
    return new SequentialCommandGroup(
      lift.setLiftBrake(),
      manipulator.setWheelSpeed(0.6),
      lift.setLiftAngle(-37, -1),
      manipulator.setManipulatorAngle(-90, -1),
      lift.setLiftPosition(2711.0),
      // The following three commands can be commented out if the manipulator's linebreak sensor isn't working
      manipulator.intakeCoral(),
      new WaitCommand(0.4),
      goToStoragePosition()
    ).withName("IntakeCoralFromStationCoralDist");
  }

  /** Intake coral from the intake station during our auto routines. Mostly identical to the first intake command
   * but also includes a 6-second timeout (it's so long because it is also executing while the robot drives to the
   * station) to make sure the bot doesn't get stuck at the station
   */
  public Command intakeCoralFromStationAuto() {
    return new SequentialCommandGroup(
      lift.setLiftBrake(),
      manipulator.setWheelSpeed(0.6),
      lift.setLiftAngle(-31, -1),
      manipulator.setManipulatorAngle(Constants.ManipulatorConstants.MinManipulatorAngle, -1),
      lift.setLiftPosition(2260.0),
      // The following command can be commented out if the manipulator's linebreak sensor isn't working
      manipulator.intakeCoral(),
      new WaitCommand(0.4)
    ).withTimeout(Seconds.of(6)).withName("IntakeCoralFromStationAuto");
  }

  /** A command for stopping the intake wheels and returning the lift to the storage position. Possibly
   * useful in the autos as a fallback command in case the manipulator's linebreak sensor fails */
  public Command finishIntake() {
    return new SequentialCommandGroup(
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("FinishIntake");
  }

  /** Our full automatic coral delivery sequence. Drive to the correct position, go to the delivery height,
   * eject the coral, and return the lift to the storage position */
  public Command deliverCoralSelectedHeight() {
    return new SequentialCommandGroup(
      deliverToAprilTag(),
      goToSelectedCoralDeliveryHeight(),
      ejectCoralL2()
    ).withName("DeliverCoralSelectedHeight");
  }

  /** Move the lift to the correct delivery position given the selected coral delivery height. The 'stopWheels'
   * command on 136 is to support when the driver gets antsy and goes directly from the intake command to this one */
  public Command goToSelectedCoralDeliveryHeight() {
    return new SequentialCommandGroup(
      manipulator.stopWheels(),
      new SelectCommand<ReefHeight>(
        Map.ofEntries(
          Map.entry(ReefHeight.L1, goToL1DeliveryPosition()),
          Map.entry(ReefHeight.L2, goToL2DeliveryPosition()),
          Map.entry(ReefHeight.L3, goToL3DeliveryPosition()),
          Map.entry(ReefHeight.L4, goToL4DeliveryPosition())
        ),
        robotcontrol::getCoralDeliveryHeight
      )
    ).withName("GoToSelectedCoralDeliveryHeight");
  }

  /** Put the lift in the correct position to deliver to L1 */
  public Command goToL1DeliveryPosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOnLiftCameraLights(),
      lift.setLiftAngle(-46, 30),
      new ParallelCommandGroup(
        lift.setLiftPosition(Constants.LiftConstants.MinLiftPos),
        manipulator.setManipulatorAngle(-90)
      )).withName("GoToL1DeliveryPosition");
  }

  /** Put the lift in the correct position to deliver to L2 */
  public Command goToL2DeliveryPosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOnLiftCameraLights(),
      manipulator.setWheelSpeed(0.05),
      lift.setLiftAngle(0, 30),
      new ParallelCommandGroup(
        lift.setLiftPosition(588),
        manipulator.setManipulatorAngle(70)
      )
    ).withName("GoToL2DeliveryPosition");
  }

  /** Put the lift in the correct position to deliver to L3 */
  public Command goToL3DeliveryPosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOnLiftCameraLights(),
      manipulator.setWheelSpeed(0.05),
      lift.setLiftAngle(0, 30),
      new ParallelCommandGroup(
        lift.setLiftPosition(2975.0),
        manipulator.setManipulatorAngle(60)
      )
    ).withName("GoToL3DeliveryPosition");
  }

  /** Put the lift in the correct position to deliver to L4 */
  public Command goToL4DeliveryPosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOnLiftCameraLights(),
      manipulator.setWheelSpeed(0.05),
      lift.setLiftAngle(-1, 40),
      new ParallelCommandGroup(
        lift.setLiftPosition(6600.0, 150),
        manipulator.setManipulatorAngle(57, 10)
      )
    ).withTimeout(Seconds.of(4)).withName("GoToL4DeliveryPosition");
  }

  /** Ejects the coral from the manipulator, assuming that we're in the L1 delivery position */
  public Command ejectCoralL1() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.4),
      new WaitCommand(0.3),
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("EjectCoralL1");
  }

  /** Ejects the coral from the manipulator, assuming that we're in the L1 delivery position */
  public Command ejectCoralL1Auto() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.4),
      new WaitCommand(0.3),
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("EjectCoralL1Auto");
  }

  /** Ejects the coral from the manipulator, assuming that we're in the L2 delivery position */
  public Command ejectCoralL2() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.3),
      new WaitCommand(0.3),
      lift.setLiftAngle(-30, -1),
      lift.setLiftPosition(0),
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("EjectCoralL2");
  }

  /** Ejects the coral from the manipulator, assuming that we're in the L3 delivery position */
  public Command ejectCoralL3() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.3),
      new WaitCommand(0.5),
      lift.setLiftAngle(-20, -1),
      lift.setLiftPosition(2575.0),
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("EjectCoralL3");
  }

  /** Ejects the coral from the manipulator, assuming that we're in the L4 delivery position */
  public Command ejectCoralL4() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.3),
      manipulator.setManipulatorAngle(80, -1),
      new WaitCommand(.4),
      // The following command provides the "flicking" mechanism that helps get
      // the coral out if it is stuck (indicated by the linebreak being tripped)
      // new DeferredCommand(
      //   () -> {
      //     if (manipulator.getLinebreakIsTripped()) {
      //       return new SequentialCommandGroup(
      //         manipulator.setManipulatorAngle(0, -1),
      //         lift.setLiftAngle(-10, 5)
      //       );
      //     } else {
      //       return new InstantCommand();
      //     }
      //   }, Set.of(RobotContainer.manipulator, RobotContainer.lift)),
      lift.setLiftAngle(-20, -1),
      lift.setLiftPosition(6400.0),
      manipulator.stopWheels(),
      goToStoragePosition()
    ).withName("EjectCoralL4");
  }

  /** Identical to the above command except it doesn't return the arm to the storage position.
   * Used during our auto routines where we drop the arm as we're moving along a path
   */
  public Command ejectCoralL4Auto() {
    return new SequentialCommandGroup(
      new WaitCommand(0.6),
      liftCamera.turnOffLiftCameraLights(),
      manipulator.setWheelSpeed(-0.3),
      manipulator.setManipulatorAngle(80, -1),
      new WaitCommand(.4),
      // The following command provides the "flicking" mechanism that helps get
      // the coral out if it is stuck (indicated by the linebreak being tripped)
      // new DeferredCommand(
      //   () -> {
      //     if (manipulator.getLinebreakIsTripped()) {
      //       return new SequentialCommandGroup(
      //         manipulator.setManipulatorAngle(0, -1),
      //         lift.setLiftAngle(-10, 5)
      //       );
      //     } else {
      //       return new InstantCommand();
      //     }
      //   }, Set.of(RobotContainer.manipulator, RobotContainer.lift)),
      lift.setLiftAngle(-20, -1),
      lift.setLiftPosition(6400.0),
      manipulator.stopWheels()
    ).withName("EjectCoralL4Auto");
  }

  /** Put the lift and manipulator in the storage position. The lift is fully retracted, straight
   * up and down, and the manipulator is holding the coral parallel to the floor */
  public Command goToStoragePosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.stopWheels(),
      manipulator.setManipulatorAngle(-75, -1),
      lift.setLiftPosition(Constants.LiftConstants.MinLiftPos, 700),
      lift.setLiftAngle(-25),
      lift.setLiftPosition(Constants.LiftConstants.MinLiftPos),
      lift.setLiftBrake(),
      lift.stopLiftMotors()
    ).withName("GoToStoragePosition");
  }

  /** Positions the lift/manipulator so that we can score on L1 during our auto routines */
  public Command positionL1AutoScore() {
    return new SequentialCommandGroup(
      lift.setLiftAngle(-40, -1),
      new ParallelCommandGroup(
        lift.setLiftPosition(600.0, -1),
        manipulator.setManipulatorAngle(-90)
      )).withName("PositionL1AutoScore");
  }

  /** Positions the lift/manipulator so that we can score on L1 with the coral sushi-style during our auto routines */
  public Command positionL1AutoScoreSushi() {
    return new SequentialCommandGroup(
      lift.setLiftAngle(-50, -1),
      new ParallelCommandGroup(
        lift.setLiftPosition(Constants.LiftConstants.MinLiftPos, -1),
        manipulator.setManipulatorAngle(0)
      )).withName("PositionL1AutoScoreSushi");
  }

  /** Stow the lift and manipulator in a compact position to allow us maximum mobility while playing defense */
  public Command goToDefensePosition() {
    return new SequentialCommandGroup(
      liftCamera.turnOffLiftCameraLights(),
      manipulator.stopWheels(),
      new ParallelCommandGroup(
        lift.setLiftPosition(Constants.LiftConstants.MinLiftPos, 700),
        manipulator.setManipulatorAngle(-98, 30)
      ),
      lift.setLiftAngle(-54),
      lift.setLiftAngleBrake(),
      lift.stopLiftAngleMotors(),
      lift.setLiftBrake(),
      lift.stopLiftMotors(),
      manipulator.setManipulatorAngleBrake(),
      manipulator.stopManipulatorAngleMotor()
    ).withName("GoToDefensePosition");
  }

  /** Move the lift/manipulator out of our compact defensive position */
  public Command exitDefensePosition() {
    return new SequentialCommandGroup(
      manipulator.stopWheels(),
      lift.setLiftAngle(0),
      new ParallelCommandGroup(
        lift.setLiftPosition(Constants.LiftConstants.MinLiftPos),
        manipulator.setManipulatorAngle(0)
      )
    ).withName("ExitDefensePosition");
  }

  /** Though this command does nothing different than the method on the manipulator, we put it here
   * so that it automatically gets registered as a named command we can call in PathPlanner
   */
  public Command stopIntakeWheels() {
    return new InstantCommand(() -> manipulator.stopWheels(), manipulator).withName("StopIntakeWheels");
  }

  /** Stop all motors from moving. Used as a safety mechanism in case something
   * breaks and we want to stop the bot from damaging itself further
   */
  public Command stopAllMotors() {
    return new SequentialCommandGroup(
      lift.stopLiftAngleMotors(),
      lift.stopLiftMotors(),
      manipulator.stopManipulatorAngleMotor(),
      manipulator.stopWheels()
    );
  }
}
