package frc.robot.commands;

import static edu.wpi.first.units.Units.Degrees;
import static edu.wpi.first.units.Units.Meters;
import static edu.wpi.first.units.Units.MetersPerSecond;
import static edu.wpi.first.units.Units.MetersPerSecondPerSecond;

import com.ctre.phoenix6.swerve.SwerveRequest;

import edu.wpi.first.math.controller.ProfiledPIDController;
import edu.wpi.first.math.geometry.Pose2d;
import edu.wpi.first.math.geometry.Pose3d;
import edu.wpi.first.math.geometry.Rotation2d;
import edu.wpi.first.math.trajectory.TrapezoidProfile.Constraints;
import edu.wpi.first.math.trajectory.TrapezoidProfile.State;
import edu.wpi.first.units.measure.Angle;
import edu.wpi.first.units.measure.Distance;
import edu.wpi.first.units.measure.LinearAcceleration;
import edu.wpi.first.units.measure.LinearVelocity;
import edu.wpi.first.wpilibj2.command.Command;
import frc.robot.Constants;
import frc.robot.Constants.ReefSide;
import frc.robot.subsystems.RobotControl;
import frc.robot.subsystems.drivetrain.CommandSwerveDrivetrain;
import frc.robot.subsystems.vision.BaseCameras;
import frc.robot.subsystems.vision.LimelightHelpers;

public class AlignToAprilTag extends Command {

  private CommandSwerveDrivetrain drivetrain;
  private BaseCameras baseCameras;
  private RobotControl robotControl;

  private static double translationP = 2.0;
  private static double translationI = 0.01;
  private static double translationD = 0.0;
  private static LinearVelocity kMaxVelocity = MetersPerSecond.of(2.0);
  private static LinearAcceleration kMaxAccel = MetersPerSecondPerSecond.of(2.0);
  private static Constraints kConstraints = new Constraints(kMaxVelocity.in(MetersPerSecond),
    kMaxAccel.in(MetersPerSecondPerSecond));
  private static Distance kTranslationTolerance = Meters.of(0.01);
  private Angle targetAngle;

  private ProfiledPIDController pidX = new ProfiledPIDController(translationP, translationI, translationD,
    kConstraints);
  private ProfiledPIDController pidY = new ProfiledPIDController(translationP, translationI, translationD,
    kConstraints);

  public AlignToAprilTag(CommandSwerveDrivetrain drivetrain, BaseCameras cameras, RobotControl robotControl) {
    this.drivetrain = drivetrain;
    this.baseCameras = cameras;
    this.robotControl = robotControl;
    pidX.setTolerance(kTranslationTolerance.in(Meters));
    pidY.setTolerance(kTranslationTolerance.in(Meters));
    addRequirements(drivetrain, baseCameras, robotControl);
  }

  // Called when the command is initially scheduled.
  @Override
  public void initialize() {
    pidX.reset(0.0);
    pidY.reset(0.0);
  }

  // Called every time the scheduler runs while the command is scheduled.
  @Override
  public void execute() {
    Pose2d aprilTagPose = robotControl.getAutoDeliveryPose();
    Pose3d targetPose = LimelightHelpers.getTargetPose3d_RobotSpace(Constants.BaseCamerasConstants.LeftLiftCameraName);
    if (targetPose == null || (targetPose.getX() == 0 && targetPose.getY() == 0)) {
      targetPose = LimelightHelpers.getTargetPose3d_RobotSpace(Constants.BaseCamerasConstants.RightLiftCameraName);
    }
    if (targetPose == null || (targetPose.getX() == 0 && targetPose.getY() == 0)) {
      return;
    }

    ReefSide delivery_side = robotControl.getCoralDeliverySide();

    pidX.setGoal(new State(0.5, 0.0));
    pidY.setGoal(new State(delivery_side == ReefSide.LEFT ? -0.1 : 0.1, 0.0));

    LinearVelocity speedX = MetersPerSecond.of(pidX.calculate(targetPose.getX())).unaryMinus();
    LinearVelocity speedY = MetersPerSecond.of(pidY.calculate(targetPose.getY())).unaryMinus();
    drivetrain
      .setControl(
        new SwerveRequest.RobotCentricFacingAngle()
          .withVelocityX(speedX)
          .withVelocityY(speedY)
          .withTargetDirection(aprilTagPose.getRotation().plus(new Rotation2d(180))));
  }

  // Called once the command ends or is interrupted.
  @Override
  public void end(boolean interrupted) {}

  // Returns true when the command should end.
  @Override
  public boolean isFinished() {
    return pidX.atGoal() && pidY.atGoal()
      && drivetrain.getState().Pose.getRotation().getMeasure().isNear(targetAngle, Degrees.of(3));
  }

}
