package frc.robot.commands;

import java.util.Set;

import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.DeferredCommand;
import edu.wpi.first.wpilibj2.command.ParallelCommandGroup;
import edu.wpi.first.wpilibj2.command.SequentialCommandGroup;
import edu.wpi.first.wpilibj2.command.WaitCommand;
import frc.robot.Constants;
import frc.robot.subsystems.Lift;
import frc.robot.subsystems.Manipulator;

public class AlgaeCommands {
  private Lift lift;
  private Manipulator manipulator;

  public AlgaeCommands(Lift lift, Manipulator manipulator) {
    this.lift = lift;
    this.manipulator = manipulator;
  }

  /** Put the lift/manipulator in the correct positions to remove algae from between the L2 and L3 reef heights */
  public Command goToLowerAlgaeRemovalPosition() {
    return new SequentialCommandGroup(
      manipulator.setWheelSpeed(0.4),
      lift.setLiftAngle(.87, -1),
      new ParallelCommandGroup(
        lift.setLiftPosition(600.0, -1),
        manipulator.setManipulatorAngle(49
        )
      )).withName("GoToLowerAlgaeRemovalPosition");
  }

  /** Put the lift/manipulator in the correct positions to remove algae from between the L3 and L4 reef heights */
  public Command goToUpperAlgaeRemovalPosition() {
    return new SequentialCommandGroup(
      manipulator.setWheelSpeed(0.4),
      lift.setLiftAngle(2, -1),
      new ParallelCommandGroup(
        lift.setLiftPosition(3400.0, -1),
        manipulator.setManipulatorAngle(45)
      )).withName("GoToUpperAlgaeRemovalPosition");
  }

  /** Put the lift/manipulator in the correct positions to pick up algae from the ground */
  public Command goToAlgaeGroundIntakePosition() {
    return new SequentialCommandGroup(
      manipulator.setWheelSpeed(0.4),
      lift.setLiftAngle(-82, -1),
      lift.setLiftPosition(1165.0, -1),
      manipulator.setManipulatorAngle(-88.5)
    ).withName("GoToAlgaeGroundIntakePosition");
  }

  /** Scores algae into the barge */
  public Command scoreInBarge() {
    return new SequentialCommandGroup(
      manipulator.setWheelSpeed(0.4),
      lift.setLiftAngle(0, 15),
      manipulator.setManipulatorAngle(-7, 20),
      lift.setLiftPositionBargeScore(6600, 1200),
      manipulator.setWheelSpeed(-0.8),
      new WaitCommand(0.25),
      manipulator.stopWheels(),
      lift.setLiftPosition(0),
      goToAlgaeStoragePosition()
    ).withName("ScoreInBarge");
  }

  /** Scores algae into the barge during autonomous (doesn't got to storage position afterwards) */
  public Command scoreInBargeAuto() {
    return new SequentialCommandGroup(
      manipulator.setWheelSpeed(0.4),
      lift.setLiftAngle(0, 15),
      manipulator.setManipulatorAngle(-7, 20),
      lift.setLiftPositionBargeScore(6600, 1200),
      manipulator.setWheelSpeed(-0.9),
      new WaitCommand(0.25),
      manipulator.stopWheels(),
      lift.setLiftPosition(3000)
    ).withName("ScoreInBargeAuto");
  }

  /** Get the lift/manipulator ready to score into the processor */
  public Command goToProcessorScoringPosition() {
    return new SequentialCommandGroup(
      lift.setLiftPosition(Constants.LiftConstants.MinLiftPos),
      new ParallelCommandGroup(
        lift.setLiftAngle(-61),
        manipulator.setManipulatorAngle(-90)
      )
    ).withName("GoToProcessorScoringPosition");
  }

  /** Score the algae into the processor */
  public Command scoreAlgaeInProcessor() {
    return new SequentialCommandGroup(
      // manipulator.setManipulatorAngle(-90, 10),
      manipulator.setWheelSpeed(-0.4),
      new WaitCommand(0.25),
      manipulator.stopWheels(),
      goToAlgaeStoragePosition()
    ).withName("ScoreAlgaeInProcessor");
  }

  /** Put the lift/manipulator in positions to hold onto the algae */
  public Command goToAlgaeStoragePosition() {
    return new DeferredCommand(
      () -> {
        double curPos = lift.getLiftPosition();
        double curManipulatorAngle = manipulator.getManipulatorAngle();
        return new SequentialCommandGroup(
          manipulator.setWheelSpeed(0.8),
          new ParallelCommandGroup(
            lift.setLiftPosition(curPos + 400, -1),
            manipulator.setManipulatorAngle(curManipulatorAngle + 10, -1)
          ),
          lift.setLiftAngle(-30, 10),
          manipulator.setManipulatorAngle(0),
          lift.setLiftPosition(Constants.LiftConstants.MinLiftPos),
          manipulator.setWheelSpeed(0.4)
        );
      }, Set.of(manipulator, lift)).withName("GoToAlgaeStoragePosition");
  }
}
