// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot;

import frc.robot.Constants.ReefFace;
import frc.robot.Constants.ReefHeight;
import frc.robot.Constants.ReefSide;
import frc.robot.commands.AlgaeCommands;
import frc.robot.commands.ClimbCommands;
import frc.robot.commands.CoralCommands;
import frc.robot.subsystems.Manipulator;
import frc.robot.subsystems.RobotControl;
import frc.robot.subsystems.RobotControl.AlgaeDeliveryMode;
import frc.robot.subsystems.RobotControl.AlgaeIntakeMode;
import frc.robot.subsystems.RobotControl.DriveMode;
import frc.robot.subsystems.drivetrain.CommandSwerveDrivetrain;
import frc.robot.subsystems.drivetrain.TunerConstants;
import frc.robot.subsystems.LEDs;
import frc.robot.subsystems.Lift;
import frc.robot.subsystems.vision.AprilTags;
import frc.robot.subsystems.vision.BaseCameras;
import frc.robot.subsystems.vision.LiftCamera;

import static edu.wpi.first.units.Units.MetersPerSecond;
import static edu.wpi.first.units.Units.RadiansPerSecond;
import static edu.wpi.first.units.Units.RotationsPerSecond;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;

import com.ctre.phoenix6.swerve.SwerveModule.DriveRequestType;
import com.ctre.phoenix6.swerve.SwerveRequest;
import com.pathplanner.lib.auto.AutoBuilder;
import com.pathplanner.lib.auto.NamedCommands;

import edu.wpi.first.wpilibj.smartdashboard.SendableChooser;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.CommandScheduler;
import edu.wpi.first.wpilibj2.command.DeferredCommand;
import edu.wpi.first.wpilibj2.command.InstantCommand;
import edu.wpi.first.wpilibj2.command.ParallelCommandGroup;
import edu.wpi.first.wpilibj2.command.SelectCommand;
import edu.wpi.first.wpilibj2.command.button.CommandJoystick;

/**
 * This class is where the bulk of the robot should be declared. Since Command-based is a
 * "declarative" paradigm, very little robot logic should actually be handled in the {@link Robot}
 * periodic methods (other than the scheduler calls). Instead, the structure of the robot (including
 * subsystems, commands, and trigger mappings) should be declared here.
 */
public class RobotContainer {
  public final static AprilTags aprilTags = new AprilTags();
  // Initialize robot subsystems
  public final static BaseCameras baseCameras = new BaseCameras();
  public final static LiftCamera liftCamera = new LiftCamera();
  public final static LEDs leds = new LEDs();
  public final static Lift lift = new Lift();
  public final static Manipulator manipulator = new Manipulator();
  public final static RobotControl robotcontrol = new RobotControl();

  // Set up bindings for necessary control of the swerve drive platform
  public final static CommandSwerveDrivetrain drivetrain = TunerConstants.createDrivetrain();
  public final static double MaxDriveSpeed = TunerConstants.kSpeedAt12Volts.in(MetersPerSecond); // Desired top translation speed
  public final static double MaxDriveAngularRate = RotationsPerSecond.of(3).in(RadiansPerSecond);
  private final SwerveRequest.FieldCentric drive = new SwerveRequest.FieldCentric()
    .withDeadband(MaxDriveSpeed * 0.05) // Add a 5% translation deadband
    .withRotationalDeadband(MaxDriveAngularRate * 0.05) // Add a 5% rotational deadband
    .withDriveRequestType(DriveRequestType.OpenLoopVoltage); // Use open-loop control for drive motors

  // Initialize multi-subsystem command factories
  public final static AlgaeCommands algaeCommands = new AlgaeCommands(lift, manipulator);
  public final static CoralCommands coralCommands = new CoralCommands(lift, manipulator, robotcontrol, liftCamera);
  public final static ClimbCommands climbCommands = new ClimbCommands(lift, liftCamera, manipulator);

  // Set up telemetry
  private final Telemetry logger = new Telemetry(MaxDriveSpeed);

  // Set up autos
  private final SendableChooser<Command> auto_chooser;

  private static final CommandSwitchController driver_controller = new CommandSwitchController(0);
  // The operator console shows up as 2 separate joysticks in the driver station, so we need to have them both here
  private static final CommandJoystick operator_controller_1 = new CommandJoystick(1);
  private static final CommandJoystick operator_controller_2 = new CommandJoystick(2);

  // The container for the robot. Contains subsystems, OI devices, and commands.
  public RobotContainer() {
    configureNamedAutoCommands(algaeCommands);
    configureNamedAutoCommands(coralCommands);
    configureNamedAutoCommands(climbCommands);
    auto_chooser = AutoBuilder.buildAutoChooser();
    SmartDashboard.putData("Auto Mode", auto_chooser);
    SmartDashboard.putData(CommandScheduler.getInstance());
    configureDefaultCommands();
    configureTriggeredCommands();
    drivetrain.registerTelemetry(logger::telemeterize);
  }

  // Set up NamedCommands that can be used by PathPlanner
  private void configureNamedAutoCommands(Object commandFactory) {
    for (Method method : commandFactory.getClass().getMethods()) {
      try {
        var returnType = method.invoke(commandFactory);
        if (Command.class.isInstance(returnType)) {
          System.out.println("Registering named auto command " + ((Command) returnType).getName());
          NamedCommands.registerCommand(((Command) returnType).getName(), (Command) returnType);
        }
      }
      catch (Exception e) {}
    }
  }

  /** Use this method to configure default commands for each subsystem to run */
  private void configureDefaultCommands() {
    // Note that X is defined as forward according to WPILib convention,
    // and Y is defined as to the left according to WPILib convention.
    leds.setDefaultCommand(leds.showModePattern());
    drivetrain.setDefaultCommand( // Drivetrain will execute this command periodically
      drivetrain.applyRequest(() -> {
        double translation_multiplier = robotcontrol.getDriveMode() == DriveMode.SLOW ? 0.8 : 1;
        double rotation_multiplier = robotcontrol.getDriveMode() == DriveMode.SLOW ? 0.5 : 1;
        return drive
          .withVelocityX(-driver_controller.getLeftY() * MaxDriveSpeed * translation_multiplier) // Drive forward with negative Y (forward)
          .withVelocityY(-driver_controller.getLeftX() * MaxDriveSpeed * translation_multiplier) // Drive left with negative X (left)
          .withRotationalRate(-driver_controller.getRightX() * MaxDriveAngularRate * rotation_multiplier); // Drive counterclockwise with negative X (left)
      }).withName("DrivetrainListener"));
  }

  /** Use this method to define your trigger -> command mappings. */
  private void configureTriggeredCommands() {
    //region Driver Controls
    // Common command set during all modes
    driver_controller.plus().and(driver_controller.minus()).onTrue(robotcontrol.rotateRobotControlMode());
    driver_controller.home().onTrue(drivetrain.runOnce(() -> drivetrain.seedFieldCentric()));
    driver_controller.rightTrigger().whileTrue(robotcontrol.holdSlowDriveMode());
    driver_controller.leftBumper().whileTrue(manipulator.holdWheelSpeedAndEnd(0.2));
    driver_controller.dPadDown().onTrue(coralCommands.goToDefensePosition());
    driver_controller.dPadUp().onTrue(coralCommands.exitDefensePosition());
    // Command set during "coral" mode
    driver_controller.x().and(robotcontrol::controlModeIsCoral)
      .toggleOnTrue(coralCommands.intakeCoralFromStationCoralDist());
    driver_controller.y().and(robotcontrol::controlModeIsCoral).toggleOnTrue(coralCommands.intakeCoralFromStation());
    driver_controller.b().and(robotcontrol::controlModeIsCoral).and(robotcontrol::aprilTagModeIsManual)
      .onTrue(coralCommands.goToSelectedCoralDeliveryHeight());
    driver_controller.b().and(robotcontrol::controlModeIsCoral).and(robotcontrol::aprilTagModeIsAuto)
      .toggleOnTrue(coralCommands.deliverToAprilTag());
    driver_controller.a().and(robotcontrol::controlModeIsCoral).onTrue(coralCommands.goToStoragePosition());
    driver_controller.rightBumper().and(robotcontrol::controlModeIsCoral).onTrue(new SelectCommand<ReefHeight>(
      Map.ofEntries(
        Map.entry(ReefHeight.L1, coralCommands.ejectCoralL1()),
        Map.entry(ReefHeight.L2, coralCommands.ejectCoralL2()),
        Map.entry(ReefHeight.L3, coralCommands.ejectCoralL3()),
        Map.entry(ReefHeight.L4, coralCommands.ejectCoralL4())
      ),
      robotcontrol::getCoralDeliveryHeight
    ).withName("EjectCoral"));
    // Command set during "algae" mode
    driver_controller.x().and(robotcontrol::controlModeIsAlgae)
      .onTrue(algaeCommands.goToAlgaeGroundIntakePosition());
    driver_controller.y().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeIntakeMode() == AlgaeIntakeMode.LOWER)
      .onTrue(algaeCommands.goToLowerAlgaeRemovalPosition());
    driver_controller.y().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeIntakeMode() == AlgaeIntakeMode.UPPER)
      .onTrue(algaeCommands.goToUpperAlgaeRemovalPosition());
    driver_controller.b().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeDeliveryMode() == AlgaeDeliveryMode.PROCESSOR)
      .onTrue(algaeCommands.goToProcessorScoringPosition());
    driver_controller.b().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeDeliveryMode() == AlgaeDeliveryMode.BARGE)
      .onTrue(algaeCommands.scoreInBarge());
    driver_controller.a().and(robotcontrol::controlModeIsAlgae).onTrue(algaeCommands.goToAlgaeStoragePosition());
    driver_controller.rightBumper().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeDeliveryMode() == AlgaeDeliveryMode.PROCESSOR)
      .onTrue(algaeCommands.scoreAlgaeInProcessor());
    driver_controller.rightBumper().and(robotcontrol::controlModeIsAlgae)
      .and(() -> robotcontrol.getAlgaeDeliveryMode() == AlgaeDeliveryMode.BARGE)
      .whileTrue(manipulator.holdWheelSpeedAndEnd(-.7));
    // Command set during "climb" mode
    driver_controller.y().and(robotcontrol::controlModeIsClimb).onTrue(climbCommands.prepForClimb());
    driver_controller.b().and(robotcontrol::controlModeIsClimb).onTrue(climbCommands.executeClimb());
    // Command set during "manual" mode
    driver_controller.rightBumper().and(robotcontrol::controlModeIsManual)
      .toggleOnTrue(coralCommands.deliverToAprilTag());
    driver_controller.capture().and(robotcontrol::controlModeIsManual).onTrue(robotcontrol.toggleDriveMode());
    driver_controller.b().and(driver_controller.leftY(0.05)).and(robotcontrol::controlModeIsManual)
      .whileTrue(new ParallelCommandGroup(
        drivetrain.applyRequest(() -> new SwerveRequest.Idle()),
        lift.adjustLiftAngleManual(() -> -driver_controller.getLeftY())
      ).withName("ManualLiftAngle"));
    driver_controller.y().and(driver_controller.leftY(0.05)).and(robotcontrol::controlModeIsManual)
      .whileTrue(new ParallelCommandGroup(
        drivetrain.applyRequest(() -> new SwerveRequest.Idle()),
        lift.adjustLiftPosManual(() -> -driver_controller.getLeftY())
      ).withName("ManualLiftPOS"));
    driver_controller.x().and(driver_controller.leftY(0.05)).and(robotcontrol::controlModeIsManual)
      .whileTrue(new ParallelCommandGroup(
        drivetrain.applyRequest(() -> new SwerveRequest.Idle()),
        manipulator.adjustManipulatorAngleManual(() -> -driver_controller.getLeftY())
      ).withName("ManualManipulatorAngle"));
    driver_controller.a().and(robotcontrol::controlModeIsManual).onTrue(coralCommands.goToStoragePosition());
    driver_controller.rightBumper().and(robotcontrol::controlModeIsManual)
      .onTrue(new DeferredCommand(
        () -> {
          if (RobotContainer.manipulator.getWheelsAreMoving()) {
            return manipulator.stopWheels();
          } else {
            return manipulator.setWheelSpeed(0.6);
          }
        },
        Set.of(manipulator)
      ));
    //endregion

    //region Operator Controls
    // Cancel is button 1 on controller 2
    // Zero is button 2 on controller 2
    operator_controller_1.button(7).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSE, ReefSide.LEFT));
    operator_controller_1.button(8).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSE, ReefSide.RIGHT));
    operator_controller_1.button(5)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSELEFT, ReefSide.LEFT));
    operator_controller_1.button(6)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSELEFT, ReefSide.RIGHT));
    operator_controller_1.button(9)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSERIGHT, ReefSide.LEFT));
    operator_controller_1.button(10)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSERIGHT, ReefSide.RIGHT));
    operator_controller_1.button(3).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARLEFT, ReefSide.LEFT));
    operator_controller_1.button(4).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARLEFT, ReefSide.RIGHT));
    operator_controller_1.button(11)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARRIGHT, ReefSide.LEFT));
    operator_controller_1.button(12)
      .onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARRIGHT, ReefSide.RIGHT));
    operator_controller_1.button(1).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FAR, ReefSide.LEFT));
    operator_controller_1.button(2).onTrue(robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FAR, ReefSide.RIGHT));
    // During coral mode, use the reef height buttons to set delivery heights. During algae
    // mode, use them to set the algae intake positions and the algae delivery positions
    operator_controller_2.button(3).onTrue(coralCommands.stopAllMotors());
    operator_controller_2.button(12).and(robotcontrol::controlModeIsCoral)
      .onTrue(robotcontrol.setCoralDeliveryHeight(ReefHeight.L1));
    operator_controller_2.button(11).and(robotcontrol::controlModeIsCoral)
      .onTrue(robotcontrol.setCoralDeliveryHeight(ReefHeight.L2));
    operator_controller_2.button(10).and(robotcontrol::controlModeIsCoral)
      .onTrue(robotcontrol.setCoralDeliveryHeight(ReefHeight.L3));
    operator_controller_2.button(9).and(robotcontrol::controlModeIsCoral)
      .onTrue(robotcontrol.setCoralDeliveryHeight(ReefHeight.L4));
    operator_controller_2.button(12).and(robotcontrol::controlModeIsAlgae)
      .onTrue(robotcontrol.setAlgaeDeliveryModeProcessor());
    operator_controller_2.button(11).and(robotcontrol::controlModeIsAlgae)
      .onTrue(robotcontrol.setAlgaeIntakeModeLower());
    operator_controller_2.button(10).and(robotcontrol::controlModeIsAlgae)
      .onTrue(robotcontrol.setAlgaeIntakeModeUpper());
    operator_controller_2.button(9).and(robotcontrol::controlModeIsAlgae)
      .onTrue(robotcontrol.setAlgaeDeliveryModeBarge());
    operator_controller_2.button(8).onTrue(robotcontrol.setControlModeAlgae());
    operator_controller_2.button(7).onTrue(robotcontrol.setControlModeCoral());
    operator_controller_2.button(6).onTrue(robotcontrol.setControlModeClimb());
    operator_controller_2.button(5).onTrue(robotcontrol.setControlModeManual());
    operator_controller_2.button(4).onTrue(robotcontrol.toggleAprilTagMode());
    // If we don't have the operator console, also allow all of their controls to be done via Shuffleboard
    SmartDashboard.putData("CloseLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSE, ReefSide.LEFT));
    SmartDashboard.putData("CloseRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSE, ReefSide.RIGHT));
    SmartDashboard.putData("CloseLeftLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSELEFT, ReefSide.LEFT));
    SmartDashboard.putData("CloseLeftRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSELEFT, ReefSide.RIGHT));
    SmartDashboard.putData("CloseRightLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSERIGHT, ReefSide.LEFT));
    SmartDashboard.putData("CloseRightRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSERIGHT, ReefSide.RIGHT));
    SmartDashboard.putData("FarLeftLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARLEFT, ReefSide.LEFT));
    SmartDashboard.putData("FarLeftRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARLEFT, ReefSide.RIGHT));
    SmartDashboard.putData("FarRightLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARRIGHT, ReefSide.LEFT));
    SmartDashboard.putData("FarRightRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FARRIGHT, ReefSide.RIGHT));
    SmartDashboard.putData("FarLeft",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FAR, ReefSide.LEFT));
    SmartDashboard.putData("FarRight",
      robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.FAR, ReefSide.RIGHT));
    SmartDashboard.putData("SetReefHeightL1", new DeferredCommand(
      () -> {
        if (robotcontrol.controlModeIsCoral()) {
          return robotcontrol.setCoralDeliveryHeight(ReefHeight.L1);
        } else if (robotcontrol.controlModeIsAlgae()) {
          return robotcontrol.setAlgaeDeliveryModeProcessor();
        } else {
          return new InstantCommand();
        }
      },
      Set.of(RobotContainer.robotcontrol)
    ));
    SmartDashboard.putData("SetReefHeightL2", new DeferredCommand(
      () -> {
        if (robotcontrol.controlModeIsCoral()) {
          return robotcontrol.setCoralDeliveryHeight(ReefHeight.L2);
        } else if (robotcontrol.controlModeIsAlgae()) {
          return robotcontrol.setAlgaeIntakeModeLower();
        } else {
          return new InstantCommand();
        }
      },
      Set.of(RobotContainer.robotcontrol)
    ));
    SmartDashboard.putData("SetReefHeightL3", new DeferredCommand(
      () -> {
        if (robotcontrol.controlModeIsCoral()) {
          return robotcontrol.setCoralDeliveryHeight(ReefHeight.L3);
        } else if (robotcontrol.controlModeIsAlgae()) {
          return robotcontrol.setAlgaeIntakeModeUpper();
        } else {
          return new InstantCommand();
        }
      },
      Set.of(RobotContainer.robotcontrol)
    ));
    SmartDashboard.putData("SetReefHeightL4", new DeferredCommand(
      () -> {
        if (robotcontrol.controlModeIsCoral()) {
          return robotcontrol.setCoralDeliveryHeight(ReefHeight.L4);
        } else if (robotcontrol.controlModeIsAlgae()) {
          return robotcontrol.setAlgaeDeliveryModeBarge();
        } else {
          return new InstantCommand();
        }
      },
      Set.of(RobotContainer.robotcontrol)
    ));
    SmartDashboard.putData("SetAlgaeMode", robotcontrol.setControlModeAlgae());
    SmartDashboard.putData("SetCoralMode", robotcontrol.setControlModeCoral());
    SmartDashboard.putData("SetClimbMode", robotcontrol.setControlModeClimb());
    SmartDashboard.putData("SetManualMode", robotcontrol.setControlModeManual());
    SmartDashboard.putData("ToggleAprilTagMode", robotcontrol.toggleAprilTagMode());
    //endregion
  }

  /** Use this to pass the autonomous command to the main {@link Robot} class.
   * @return the command to run in autonomous */
  public Command getAutonomousCommand() {
    return auto_chooser.getSelected();
  }
}
