// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

package frc.robot;

import com.ctre.phoenix6.Utils;
import com.pathplanner.lib.commands.FollowPathCommand;

import edu.wpi.first.wpilibj.DataLogManager;
import edu.wpi.first.wpilibj.DriverStation;
import edu.wpi.first.wpilibj.RobotController;
import edu.wpi.first.wpilibj.TimedRobot;
import edu.wpi.first.wpilibj.DriverStation.Alliance;
import edu.wpi.first.wpilibj.smartdashboard.SmartDashboard;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.CommandScheduler;
import frc.robot.Constants.ReefFace;
import frc.robot.Constants.ReefSide;

/**
 * The methods in this class are called automatically corresponding to each
 * mode, as described in the TimedRobot documentation. If you change the
 * name of this class or the package after creating this project, you must
 * also update the Main.java file in the project.
 */
public class Robot extends TimedRobot {
  private Command autonomous_command;
  private Command led_command;
  private final RobotContainer robot_container;

  /** This function is run when the robot is first started up and should be used
   * for any initialization code. */
  public Robot() {
    // Instantiate our RobotContainer. This will perform all our button bindings,
    // and put our autonomous chooser on the dashboard.
    robot_container = new RobotContainer();
    FollowPathCommand.warmupCommand().withName("WarmupPathplanner").schedule();
    RobotContainer.robotcontrol.setCoralDeliveryFaceAndSide(ReefFace.CLOSE, ReefSide.LEFT).schedule();
    DataLogManager.start();
    DriverStation.startDataLog(DataLogManager.getLog());
  }

  /** This function is called every 20 ms, no matter the mode. Use this for items like diagnostics
   * that you want ran during disabled, autonomous, teleoperated and test.
   * <p> This runs after the mode specific periodic functions, but before LiveWindow and
   * SmartDashboard integrated updating. */
  @Override
  public void robotPeriodic() {
    // Runs the Scheduler. This is responsible for polling buttons, adding newly-scheduled
    // commands, running already-scheduled commands, removing finished or interrupted commands,
    // and running subsystem periodic() methods. This must be called from the robot's periodic
    // block in order for anything in the Command-based framework to work.
    CommandScheduler.getInstance().run();
    DashboardLinkedNumber.checkAll();
    if (!Utils.isSimulation()) {
      SmartDashboard.putNumber("BatteryVoltage", RobotController.getBatteryVoltage());
    }
  }

  /** This function is called once each time the robot enters Disabled mode. */
  @Override
  public void disabledInit() {
    led_command = RobotContainer.leds.breathAllyColor();
    led_command.schedule();
  }

  @Override
  public void disabledPeriodic() {}

  /** This autonomous runs the autonomous command selected by your {@link RobotContainer} class. */
  @Override
  public void autonomousInit() {
    led_command = RobotContainer.leds.solidAllyColor();
    led_command.schedule();

    DriverStation.getAlliance().ifPresent(alliance -> {
      if (alliance == Alliance.Blue) {
        RobotContainer.drivetrain.getPigeon2().setYaw(180);
      }
    });

    autonomous_command = robot_container.getAutonomousCommand();
    // schedule the autonomous command (example)
    if (autonomous_command != null) {
      autonomous_command.schedule();
    }
  }

  /** This function is called periodically during autonomous. */
  @Override
  public void autonomousPeriodic() {}

  @Override
  public void teleopInit() {
    if (led_command != null) {
      led_command.cancel();
    }

    // This makes sure that the autonomous stops running when
    // teleop starts running. If you want the autonomous to
    // continue until interrupted by another command, remove
    // this line or comment it out.
    if (autonomous_command != null) {
      autonomous_command.cancel();
    }
  }

  /** This function is called periodically during operator control. */
  @Override
  public void teleopPeriodic() {}

  @Override
  public void testInit() {
    // Cancels all running commands at the start of test mode.
    CommandScheduler.getInstance().cancelAll();
  }

  /** This function is called periodically during test mode. */
  @Override
  public void testPeriodic() {}

  /** This function is called once when the robot is first started up. */
  @Override
  public void simulationInit() {}

  /** This function is called periodically whilst in simulation. */
  @Override
  public void simulationPeriodic() {}
}
