package frc.robot;

import edu.wpi.first.wpilibj.GenericHID;
import edu.wpi.first.wpilibj.event.EventLoop;
import edu.wpi.first.wpilibj2.command.CommandScheduler;
import edu.wpi.first.wpilibj2.command.button.*;

/** A Command-based controller class for the Spectra Nintendo Switch controller */
public class CommandSwitchController extends CommandGenericHID {
  private final GenericHID hid;

  // Represents an axis on a SwitchController
  public enum Axis {
    LEFTX(0), // Left X axis
    LEFTY(1), // Left Y axis
    RIGHTX(2), // Right X axis
    RIGHTY(3); // Right Y axis

    public final int value; // Axis value

    Axis(int value) {
      this.value = value;
    }
  }

  // Represents a button on a SwitchController
  public enum Button {
    Y(1), // Y Button
    B(2), // B Button
    A(3), // A Button
    X(4), // X Button
    LEFTBUMPER(5), // Left bumper Button
    RIGHTBUMPER(6), // Right bumper Button
    LEFTTRIGGER(7), // Left trigger Button
    RIGHTTRIGGER(8), // Right trigger Button
    MINUS(9), // Minus Button
    PLUS(10), // Plus Button
    LEFTSTICK(11), // Left stick Button
    RIGHTSTICK(12), // Right stick Button
    HOME(13), // Home Button
    CAPTURE(14); // Capture Button

    public final int value; // Axis value

    Button(int value) {
      this.value = value;
    }
  }

  public CommandSwitchController(int port) {
    super(port);
    hid = new GenericHID(port);
  }

  public boolean aIsPressed() {
    return hid.getRawButton(Button.A.value);
  }

  public Trigger a() {
    return a(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger a(EventLoop loop) {
    return button(Button.A.value, loop);
  }

  public boolean bIsPressed() {
    return hid.getRawButton(Button.B.value);
  }

  public Trigger b() {
    return b(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger b(EventLoop loop) {
    return button(Button.B.value, loop);
  }

  public boolean xIsPressed() {
    return hid.getRawButton(Button.X.value);
  }

  public Trigger x() {
    return x(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger x(EventLoop loop) {
    return button(Button.X.value, loop);
  }

  public boolean yIsPressed() {
    return hid.getRawButton(Button.Y.value);
  }

  public Trigger y() {
    return y(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger y(EventLoop loop) {
    return button(Button.Y.value, loop);
  }

  public boolean leftBumperIsPressed() {
    return hid.getRawButton(Button.LEFTBUMPER.value);
  }

  public Trigger leftBumper() {
    return leftBumper(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger leftBumper(EventLoop loop) {
    return button(Button.LEFTBUMPER.value, loop);
  }

  public boolean rightBumperIsPressed() {
    return hid.getRawButton(Button.RIGHTBUMPER.value);
  }

  public Trigger rightBumper() {
    return rightBumper(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger rightBumper(EventLoop loop) {
    return button(Button.RIGHTBUMPER.value, loop);
  }

  public boolean leftTriggerIsPressed() {
    return hid.getRawButton(Button.LEFTTRIGGER.value);
  }

  public Trigger leftTrigger() {
    return leftTrigger(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger leftTrigger(EventLoop loop) {
    return button(Button.LEFTTRIGGER.value, loop);
  }

  public boolean rightTriggerIsPressed() {
    return hid.getRawButton(Button.RIGHTTRIGGER.value);
  }

  public Trigger rightTrigger() {
    return rightTrigger(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger rightTrigger(EventLoop loop) {
    return button(Button.RIGHTTRIGGER.value, loop);
  }

  public boolean minusIsPressed() {
    return hid.getRawButton(Button.MINUS.value);
  }

  public Trigger minus() {
    return minus(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger minus(EventLoop loop) {
    return button(Button.MINUS.value, loop);
  }

  public boolean plusIsPressed() {
    return hid.getRawButton(Button.PLUS.value);
  }

  public Trigger plus() {
    return plus(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger plus(EventLoop loop) {
    return button(Button.PLUS.value, loop);
  }

  public boolean leftStickIsPressed() {
    return hid.getRawButton(Button.LEFTSTICK.value);
  }

  public Trigger leftStick() {
    return leftStick(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger leftStick(EventLoop loop) {
    return button(Button.LEFTSTICK.value, loop);
  }

  public boolean rightStickIsPressed() {
    return hid.getRawButton(Button.RIGHTSTICK.value);
  }

  public Trigger rightStick() {
    return rightStick(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger rightStick(EventLoop loop) {
    return button(Button.RIGHTSTICK.value, loop);
  }

  public boolean homeIsPressed() {
    return hid.getRawButton(Button.HOME.value);
  }

  public Trigger home() {
    return home(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger home(EventLoop loop) {
    return button(Button.HOME.value, loop);
  }

  public boolean captureIsPressed() {
    return hid.getRawButton(Button.CAPTURE.value);
  }

  public Trigger capture() {
    return capture(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger capture(EventLoop loop) {
    return button(Button.CAPTURE.value, loop);
  }

  public Trigger dPadUp() {
    return dPadUp(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadUp(EventLoop loop) {
    return pov(0);
  }

  public Trigger dPadUpRight() {
    return dPadUpRight(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadUpRight(EventLoop loop) {
    return pov(45);
  }

  public Trigger dPadRight() {
    return dPadRight(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadRight(EventLoop loop) {
    return pov(90);
  }

  public Trigger dPadDownRight() {
    return dPadDownRight(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadDownRight(EventLoop loop) {
    return pov(135);
  }

  public Trigger dPadDown() {
    return dPadDown(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadDown(EventLoop loop) {
    return pov(180);
  }

  public Trigger dPadDownLeft() {
    return dPadDownLeft(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadDownLeft(EventLoop loop) {
    return pov(225);
  }

  public Trigger dPadLeft() {
    return dPadLeft(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadLeft(EventLoop loop) {
    return pov(270);
  }

  public Trigger dPadUpLeft() {
    return dPadUpLeft(CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  public Trigger dPadUpLeft(EventLoop loop) {
    return pov(315);
  }

  public double getLeftX() {
    return hid.getRawAxis(Axis.LEFTX.value);
  }

  /** A trigger that fires when the left X axis goes outside the deadband */
  public Trigger leftX(double deadband) {
    return leftX(deadband, CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  private Trigger leftX(double deadband, EventLoop loop) {
    return axisMagnitudeGreaterThan(Axis.LEFTX.value, deadband);
  }

  public double getLeftY() {
    return hid.getRawAxis(Axis.LEFTY.value);
  }

  /** A trigger that fires when the left Y axis goes outside the deadband */
  public Trigger leftY(double deadband) {
    return leftY(deadband, CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  private Trigger leftY(double deadband, EventLoop loop) {
    return axisMagnitudeGreaterThan(Axis.LEFTY.value, deadband);
  }

  public double getRightX() {
    return hid.getRawAxis(Axis.RIGHTX.value);
  }

  /** A trigger that fires when the right X axis goes outside the deadband */
  public Trigger rightX(double deadband) {
    return rightX(deadband, CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  private Trigger rightX(double deadband, EventLoop loop) {
    return axisMagnitudeGreaterThan(Axis.RIGHTX.value, deadband);
  }

  public double getRightY() {
    return hid.getRawAxis(3);
  }

  /** A trigger that fires when the right Y axis goes outside the deadband */
  public Trigger rightY(double deadband) {
    return rightY(deadband, CommandScheduler.getInstance().getDefaultButtonLoop());
  }

  private Trigger rightY(double deadband, EventLoop loop) {
    return axisMagnitudeGreaterThan(Axis.RIGHTY.value, deadband);
  }
}
